#!/usr/bin/env python3
"""
DOCplex 2.29.245 Implementation for Wedding Optimization Problem
"""

from docplex.mp.model import Model

def wedding_optimization():
    """Optimize the number of weddings to minimize total cost while respecting constraints."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="wedding_optimization")
    
    # Data from the problem
    churches = [1, 2, 3]
    years = [2023, 2024]
    
    # Cost per wedding at each church and year
    cost_per_wedding = {
        (1, 2023): 5000.0,
        (2, 2023): 4500.0,
        (3, 2023): 6000.0,
        (1, 2024): 5200.0,
        (2, 2024): 4600.0,
        (3, 2024): 6100.0
    }
    
    # Maximum number of weddings per year
    max_weddings_per_year = 120
    
    # Maximum number of weddings per church per year
    max_weddings_per_church = 30
    
    # 2. VARIABLES
    # Decision variables: number of weddings at each church each year
    x = {(c, y): mdl.continuous_var(name=f"x_{c}_{y}", lb=0) for c in churches for y in years}
    
    # 3. OBJECTIVE FUNCTION
    # Minimize the total cost of organizing weddings
    total_cost = mdl.sum(cost_per_wedding[c, y] * x[c, y] for c in churches for y in years)
    mdl.minimize(total_cost)
    
    # 4. CONSTRAINTS
    
    # Annual Wedding Limit: Total weddings per year <= 120
    for y in years:
        total_weddings = mdl.sum(x[c, y] for c in churches)
        mdl.add_constraint(total_weddings <= max_weddings_per_year, ctname=f"annual_limit_{y}")
    
    # Church Capacity Limit: Weddings per church per year <= 30
    for c in churches:
        for y in years:
            mdl.add_constraint(x[c, y] <= max_weddings_per_church, ctname=f"church_capacity_{c}_{y}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for c in churches:
            for y in years:
                value = solution.get_value(x[c, y])
                if value > 1e-6:
                    print(f"Weddings at church {c} in year {y}: {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Execute the optimization
if __name__ == "__main__":
    wedding_optimization()