#!/usr/bin/env python3
"""
Pyomo 6.9.2 Implementation for Workshop Paper Selection Optimization
"""

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def workshop_paper_optimization():
    """Optimize the selection of workshop submissions to maximize quality scores."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Workshop capacities
    workshop_capacity_data = {
        1: 12,
        2: 15,
        3: 10
    }
    
    # Submission scores
    submission_scores_data = {
        1: 8.7,
        2: 7.5,
        3: 9.2
    }
    
    # Submission to workshop mapping
    submission_workshop_mapping_data = {
        1: 1,
        2: 2,
        3: 3
    }
    
    # Validate data consistency
    submission_ids = set(submission_scores_data.keys())
    workshop_ids = set(workshop_capacity_data.keys())
    mapped_submissions = set(submission_workshop_mapping_data.keys())
    mapped_workshops = set(submission_workshop_mapping_data.values())
    
    assert submission_ids == mapped_submissions, "Submission IDs mismatch in mapping"
    assert mapped_workshops.issubset(workshop_ids), "Workshop IDs mismatch in mapping"
    
    # 3. SETS
    model.SUBMISSIONS = pyo.Set(initialize=submission_ids)
    model.WORKSHOPS = pyo.Set(initialize=workshop_ids)
    
    # 4. PARAMETERS
    model.scores = pyo.Param(model.SUBMISSIONS, initialize=submission_scores_data)
    model.capacities = pyo.Param(model.WORKSHOPS, initialize=workshop_capacity_data)
    model.mapping = pyo.Param(model.SUBMISSIONS, initialize=submission_workshop_mapping_data)
    
    # 5. VARIABLES
    model.accept = pyo.Var(model.SUBMISSIONS, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.scores[i] * model.accept[i] for i in model.SUBMISSIONS)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    # Workshop capacity constraints
    def capacity_rule(model, j):
        return sum(model.accept[i] for i in model.SUBMISSIONS if model.mapping[i] == j) <= model.capacities[j]
    model.capacity_constraint = pyo.Constraint(model.WORKSHOPS, rule=capacity_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nAccepted submissions:")
        for i in model.SUBMISSIONS:
            if pyo.value(model.accept[i]) > 0.5:
                print(f"Submission {i} accepted (Score: {model.scores[i]})")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Run the optimization
if __name__ == "__main__":
    workshop_paper_optimization()