#!/usr/bin/env python3

import random
import json
import re
from config import *
from verification import extract_mathematical_formulation_from_alternating

def should_skip_database(db):
    """Check if database should be skipped (more than MAX_TABLES tables)"""
    table_count = len(db['table_names_original'])
    return table_count > MAX_TABLES

def simple_convergence_check(iteration, or_analysis):
    """Simplified convergence check"""
    if iteration >= MAX_ALTERNATING_ITERATIONS:
        return True
    iteration_status = or_analysis.get("iteration_status", {})
    if iteration_status.get("complete", False):
        return True
    return False

def create_state_info(or_analysis, implementation_result, iteration):
    """Create state.json information"""
    return {
        "iteration": iteration,
        "converged": or_analysis.get("iteration_status", {}).get("complete", False),
        "business_context": or_analysis.get("business_context", "unknown"),
        "optimization_problem": or_analysis.get("optimization_problem_description", "unknown"),
        "objective": or_analysis.get("optimization_formulation", {}).get("objective", "unknown"),
        "table_count": len(implementation_result.get("schema_adjustment_decisions", {}).get("tables_to_create", [])),
        "key_changes": [implementation_result.get("implementation_summary", "")],
        "math_consistency": "high" if implementation_result.get("validation", {}).get("math_traceable", False) else "medium",
        "next_iteration_focus": or_analysis.get("iteration_status", {}).get("next_focus", "Continue refinement"),
        "mapping_adequacy_summary": analyze_mapping_adequacy(or_analysis.get("current_optimization_to_schema_mapping", {}))
    }

def analyze_mapping_adequacy(mapping_info):
    """Analyze overall mapping adequacy from OR expert mapping"""
    adequacy_counts = {"good": 0, "missing": 0, "redundant": 0, "partial": 0, "inaccurate": 0}
    total_mappings = 0
    
    for category in ["objective_coefficients", "constraint_bounds", "decision_variables"]:
        category_mappings = mapping_info.get(category, {})
        for mapping_name, mapping_details in category_mappings.items():
            adequacy = mapping_details.get("mapping_adequacy", "unknown")
            if adequacy in adequacy_counts:
                adequacy_counts[adequacy] += 1
            total_mappings += 1
    
    if total_mappings == 0:
        return "no_mappings"
    
    good_ratio = adequacy_counts["good"] / total_mappings
    if good_ratio >= 0.8:
        return "mostly_good"
    elif good_ratio >= 0.5:
        return "partially_adequate"
    else:
        return "needs_improvement"

def create_summary_text(or_analysis, implementation_result, iteration):
    """Create human-readable summary.txt"""
    return f"""Iteration {iteration} Summary
Business Context: {or_analysis.get('business_context', 'Unknown')}
Optimization Problem: {or_analysis.get('optimization_problem_description', 'Unknown')}
Objective: {or_analysis.get('optimization_formulation', {}).get('objective', 'Unknown')}
Tables Created: {len(implementation_result.get('schema_adjustment_decisions', {}).get('tables_to_create', []))}
Tables Modified: {len(implementation_result.get('schema_adjustment_decisions', {}).get('tables_to_modify', []))}
Tables Deleted: {len(implementation_result.get('schema_adjustment_decisions', {}).get('tables_to_delete', []))}
Key Change: {implementation_result.get('implementation_summary', 'No changes')}
Status: {'Complete' if or_analysis.get('iteration_status', {}).get('complete', False) else 'In progress'}
Confidence: {or_analysis.get('iteration_status', {}).get('confidence', 'Unknown')}
Next Focus: {or_analysis.get('iteration_status', {}).get('next_focus', 'Continue')}
Mapping Adequacy: {analyze_mapping_adequacy(or_analysis.get('current_optimization_to_schema_mapping', {}))}
Business Configuration Parameters: {len(implementation_result.get('business_configuration_logic_updates', {}).get('configuration_parameters', {}))}
"""

def generate_sql_files_from_implementation(implementation_result, previous_schema_sql=""):
    """Generate schema.sql and data.sql from implementation result"""
    
    # Extract information from implementation
    schema_result = implementation_result.get("schema_adjustment_decisions", {})
    data_mapping = implementation_result.get("data_mapping", {})
    data_dictionary = implementation_result.get("data_dictionary", {})
    business_config_updates = implementation_result.get("business_configuration_logic_updates", {})
    iteration = implementation_result.get("iteration", 0)
    
    # Generate schema.sql
    schema_sql = f"""-- Iteration {iteration} Database Schema
-- Objective: {implementation_result.get('implementation_summary', 'Optimization problem')}

"""
    
    # Generate CREATE TABLE statements based on data dictionary
    tables_info = data_dictionary.get("tables", {})
    for table_name, table_info in tables_info.items():
        columns_info = table_info.get("columns", {})
        if not columns_info:
            continue
            
        schema_sql += f"""CREATE TABLE {table_name} (\n"""
        column_definitions = []
        for col_name, col_info in columns_info.items():
            data_type = col_info.get("data_type", "NUMBER")
            column_definitions.append(f"  {col_name} {data_type}")
        
        schema_sql += ",\n".join(column_definitions)
        schema_sql += f"\n);\n\n"
    
    # Fallback for tables without detailed dictionary
    tables_created = schema_result.get("tables_to_create", [])
    for table_info in tables_created:
        table_name = table_info.get("table_name", "unknown_table")
        if table_name not in tables_info:
            schema_sql += f"""CREATE TABLE {table_name} (
  id INTEGER PRIMARY KEY,
  value NUMBER
);

"""
    
    # Generate business configuration logic
    business_configuration_logic = business_config_updates.get("configuration_parameters", {})
    
    # Generate data.sql (placeholder - will be replaced by triple expert)
    data_sql = f"""-- Iteration {iteration} Sample Data
-- Will be replaced by triple expert realistic data generation

"""
    
    sample_data_rows = data_mapping.get("sample_data_rows", {})
    for table_name, row_count in sample_data_rows.items():
        table_info = tables_info.get(table_name, {})
        columns_info = table_info.get("columns", {})
        
        data_sql += f"""-- Placeholder data for {table_name} (will be replaced by triple expert)
"""
        
        if columns_info:
            column_names = list(columns_info.keys())
            for i in range(row_count):
                values = []
                for col_name in column_names:
                    col_info = columns_info[col_name]
                    sample_vals = col_info.get("sample_values", "")
                    
                    # Fix: Ensure sample_vals is a string before calling .lower()
                    if isinstance(sample_vals, list):
                        sample_vals = str(sample_vals)
                    elif sample_vals is None:
                        sample_vals = ""
                    else:
                        sample_vals = str(sample_vals)
                    
                    if "range" in sample_vals.lower() or "-" in sample_vals:
                        values.append(str(random.randint(10, 100)))
                    else:
                        values.append(str(random.randint(10, 100)))
                
                data_sql += f"-- INSERT INTO {table_name} ({', '.join(column_names)}) VALUES ({', '.join(values)});\n"
        else:
            for i in range(row_count):
                data_sql += f"-- INSERT INTO {table_name} (id, value) VALUES ({i+1}, {random.randint(10, 100)});\n"
        data_sql += "\n"
    
    return schema_sql, data_sql, business_configuration_logic

def generate_sql_files_from_triple_expert(triple_expert_result, final_schema_sql, final_business_configuration_logic, iteration):
    """Generate schema.sql, data.sql and business_configuration.json from triple expert result"""
    
    # Use the final schema as-is
    schema_sql = final_schema_sql
    
    # Generate realistic data.sql from triple expert
    generated_data = triple_expert_result.get("generated_data", {})
    business_config_values = triple_expert_result.get("business_configuration_values", {})
    
    data_sql = f"""-- Iteration {iteration} Realistic Data
-- Generated by triple expert (business + data + optimization)
-- {triple_expert_result.get('data_generation_approach', 'Realistic data generation')}

"""
    
    for table_name, rows_data in generated_data.items():
        if not rows_data:
            continue
            
        data_sql += f"""-- Realistic data for {table_name}
"""
        
        # Get column names from first row
        if rows_data and isinstance(rows_data[0], dict):
            sample_row = rows_data[0]
            columns = [col for col in sample_row.keys() if col != "business_justification"]
            
            for row_data in rows_data:
                values = []
                for col in columns:
                    value = row_data.get(col, "NULL")
                    # Handle string values that need quotes
                    if isinstance(value, str) and not value.replace('.','').replace('-','').isdigit():
                        if value.upper() != "NULL":
                            values.append(f"'{value}'")
                        else:
                            values.append(value)
                    else:
                        values.append(str(value))
                
                data_sql += f"INSERT INTO {table_name} ({', '.join(columns)}) VALUES ({', '.join(values)});\n"
            
            data_sql += "\n"
    
    # Generate final business configuration with realistic values
    final_business_configuration = {}
    
    if final_business_configuration_logic:
        for param_name, param_logic in final_business_configuration_logic.items():
            # Start with the logic template
            final_param = {
                "data_type": param_logic.get("data_type"),
                "business_meaning": param_logic.get("business_meaning"),
                "optimization_role": param_logic.get("optimization_role"),
                "configuration_type": param_logic.get("configuration_type")
            }
            
            # Handle different configuration types
            if param_logic.get("configuration_type") == "business_logic_formula":
                # Keep formula expressions unchanged
                final_param["formula_expression"] = param_logic.get("formula_expression")
            else:
                # Replace sample_value with realistic value from triple expert for scalar parameters
                if param_name in business_config_values:
                    final_param["value"] = business_config_values[param_name].get("value", param_logic.get("sample_value"))
                    final_param["business_justification"] = business_config_values[param_name].get("business_justification", "")
                else:
                    # Fallback to sample value if triple expert didn't provide one
                    final_param["value"] = param_logic.get("sample_value")
                    final_param["business_justification"] = "Using default sample value"
            
            final_business_configuration[param_name] = final_param
    
    return schema_sql, data_sql, final_business_configuration

def extract_code_from_response(response):
    """Extract Python code from LLM response"""
    
    # Clean response
    response = response.strip()
    
    # Look for code blocks
    code_pattern = r'```(?:python)?\s*(.*?)```'
    matches = re.findall(code_pattern, response, re.DOTALL)
    
    if matches:
        # Return the first (and usually only) code block
        code = matches[0].strip()
        return code
    
    # If no code blocks found, try to extract everything after certain keywords
    keywords = ["import", "from", "#"]
    for keyword in keywords:
        if keyword in response:
            lines = response.split('\n')
            code_lines = []
            start_found = False
            
            for line in lines:
                if not start_found and keyword in line:
                    start_found = True
                if start_found:
                    code_lines.append(line)
            
            if code_lines:
                return '\n'.join(code_lines)
    
    # Last resort: return the whole response if it looks like code
    if any(word in response.lower() for word in ['import', 'def ', 'class ', 'if ', 'for ', 'while ']):
        return response
    
    return ""

def check_solver_values_consistency(solver_results):
    """Check if at least two solvers succeeded with consistent values"""
    
    # Collect successful solver results
    successful_results = []
    error_statuses = ["error", "execution_error", "timeout", "code_generation_failed", "generation_error"]
    
    for solver in ["gurobipy", "docplex", "pyomo"]:
        result = solver_results.get(solver, {})
        status = result.get("status", "unknown")
        optimal_value = result.get("optimal_value")
        
        # Only consider solvers that succeeded and have optimal values
        if status == "optimal" and optimal_value is not None:
            successful_results.append({
                "solver": solver,
                "optimal_value": optimal_value,
                "status": status
            })
    
    # Need at least 2 successful solvers
    if len(successful_results) < 2:
        return False  # Not enough successful solvers
    
    # Check if any two solvers have consistent values
    for i in range(len(successful_results)):
        for j in range(i + 1, len(successful_results)):
            value1 = successful_results[i]["optimal_value"]
            value2 = successful_results[j]["optimal_value"]
            
            # If any pair is consistent, return True
            if abs(value1 - value2) <= SOLVER_CONSISTENCY_TOLERANCE:
                return True
    
    # No consistent pair found
    return False

def check_solver_consistency(solver_results):
    """Check if at least two solvers succeeded with consistent values (for summary reporting)"""
    
    # Collect successful solver results
    successful_results = []
    
    for solver in ["gurobipy", "docplex", "pyomo"]:
        result = solver_results.get(solver, {})
        status = result.get("status", "unknown")
        optimal_value = result.get("optimal_value")
        
        # Only consider solvers that succeeded and have optimal values
        if status == "optimal" and optimal_value is not None:
            successful_results.append(optimal_value)
    
    # Need at least 2 successful solvers
    if len(successful_results) < 2:
        return False
    
    # Check if any two values are consistent
    for i in range(len(successful_results)):
        for j in range(i + 1, len(successful_results)):
            if abs(successful_results[i] - successful_results[j]) <= 1e-6:
                return True
    
    return False

def extract_business_configuration_requirements(business_configuration):
    """Extract business configuration requirements for automatic prompt generation"""
    
    scalar_parameters = []
    business_formulas = []
    
    if not business_configuration:
        return scalar_parameters, business_formulas
    
    for param_name, param_info in business_configuration.items():
        config_type = param_info.get("configuration_type", "")
        business_meaning = param_info.get("business_meaning", "")
        optimization_role = param_info.get("optimization_role", "")
        
        if config_type == "scalar_parameter":
            scalar_parameters.append({
                "name": param_name,
                "meaning": business_meaning,
                "role": optimization_role,
                "value": param_info.get("value", param_info.get("sample_value"))
            })
        elif config_type == "business_logic_formula":
            business_formulas.append({
                "name": param_name,
                "meaning": business_meaning,
                "role": optimization_role,
                "formula": param_info.get("formula_expression", "")
            })
    
    return scalar_parameters, business_formulas

def format_business_requirements_for_prompt(scalar_parameters, business_formulas):
    """Format business requirements as natural language for prompt context"""
    
    requirements = []
    
    # Format scalar parameters
    if scalar_parameters:
        scalar_items = []
        for param in scalar_parameters:
            scalar_items.append(f"{param['meaning']} (used for {param['role']})")
        requirements.append(f"Business configuration includes: {', '.join(scalar_items)}")
    
    # Format business logic formulas
    if business_formulas:
        formula_items = []
        for formula in business_formulas:
            formula_items.append(f"{formula['meaning']} (calculation method for {formula['role']})")
        requirements.append(f"Business logic formulas to express in natural language: {', '.join(formula_items)}")
    
    return requirements

def format_enhanced_solver_results_for_markdown(solver_results, solver_codes, solver_analysis_result):
    """Format solver results with enhanced analysis for inclusion in markdown documentation"""
    
    markdown_sections = []
    
    # Section 5: Gurobipy Implementation
    gurobipy_result = solver_results.get("gurobipy", {})
    gurobipy_code = solver_codes.get("gurobipy", {}).get("extracted_code", "")
    gurobipy_analysis = solver_analysis_result.get("solver_analysis", {}).get("gurobipy_analysis", {})
    
    markdown_sections.append("## 5. Gurobipy Implementation")
    markdown_sections.append("")
    markdown_sections.append("```python")
    markdown_sections.append(gurobipy_code if gurobipy_code else "# Code generation failed")
    markdown_sections.append("```")
    markdown_sections.append("")
    
    # Enhanced execution results
    markdown_sections.append("### Execution Results")
    if gurobipy_result.get("status") == "optimal":
        markdown_sections.append("**Status**: OPTIMAL")
        optimal_val = gurobipy_result.get('optimal_value', 'N/A')
        if optimal_val != 'N/A':
            markdown_sections.append(f"**Optimal Value**: {optimal_val}")
        else:
            markdown_sections.append("**Optimal Value**: N/A")
        markdown_sections.append(f"**Execution Time**: {gurobipy_result.get('execution_time', 0):.2f} seconds")
        
        # Decision variables with business meaning
        decision_vars = gurobipy_result.get("decision_variables", {})
        if decision_vars:
            markdown_sections.append("**Decision Variables**:")
            for var_name, var_value in decision_vars.items():
                business_meaning = solver_analysis_result.get("business_insights", {}).get("decision_variable_meanings", {}).get(var_name, f"Resource allocation for {var_name}")
                markdown_sections.append(f"- {var_name} = {var_value:.3f} ({business_meaning})")
        
        markdown_sections.append(f"**Reliability**: {gurobipy_analysis.get('reliability', 'N/A')}")
        
        # Add retry information if available
        if gurobipy_result.get("retry_attempt"):
            markdown_sections.append(f"**Retry Attempt**: {gurobipy_result.get('retry_attempt')}")
    else:
        status = gurobipy_result.get("status", "unknown").upper()
        markdown_sections.append(f"**Status**: {status}")
        markdown_sections.append(f"**Error**: {gurobipy_result.get('error_message', 'Unknown error')}")
    
    if gurobipy_analysis.get('notes'):
        markdown_sections.append(f"**Analysis**: {gurobipy_analysis.get('notes')}")
    
    markdown_sections.append("")
    
    # Section 6: DOCplex Implementation
    docplex_result = solver_results.get("docplex", {})
    docplex_code = solver_codes.get("docplex", {}).get("extracted_code", "")
    docplex_analysis = solver_analysis_result.get("solver_analysis", {}).get("docplex_analysis", {})
    
    markdown_sections.append("## 6. DOCplex Implementation")
    markdown_sections.append("")
    markdown_sections.append("```python")
    markdown_sections.append(docplex_code if docplex_code else "# Code generation failed")
    markdown_sections.append("```")
    markdown_sections.append("")
    
    # Enhanced execution results
    markdown_sections.append("### Execution Results")
    if docplex_result.get("status") == "optimal":
        markdown_sections.append("**Status**: OPTIMAL")
        optimal_val = docplex_result.get('optimal_value', 'N/A')
        if optimal_val != 'N/A':
            markdown_sections.append(f"**Optimal Value**: {optimal_val}")
        else:
            markdown_sections.append("**Optimal Value**: N/A")
        exec_time = docplex_result.get('execution_time', 0)
        markdown_sections.append(f"**Execution Time**: {exec_time:.2f} seconds")
        
        # Decision variables with business meaning
        decision_vars = docplex_result.get("decision_variables", {})
        if decision_vars:
            markdown_sections.append("**Decision Variables**:")
            for var_name, var_value in decision_vars.items():
                business_meaning = solver_analysis_result.get("business_insights", {}).get("decision_variable_meanings", {}).get(var_name, f"Resource allocation for {var_name}")
                markdown_sections.append(f"- {var_name} = {var_value:.3f} ({business_meaning})")
        
        markdown_sections.append(f"**Reliability**: {docplex_analysis.get('reliability', 'N/A')}")
        
        # Add retry information if available
        if docplex_result.get("retry_attempt"):
            markdown_sections.append(f"**Retry Attempt**: {docplex_result.get('retry_attempt')}")
    else:
        status = docplex_result.get("status", "unknown").upper()
        markdown_sections.append(f"**Status**: {status}")
        markdown_sections.append(f"**Error**: {docplex_result.get('error_message', 'Unknown error')}")
    
    if docplex_analysis.get('notes'):
        markdown_sections.append(f"**Analysis**: {docplex_analysis.get('notes')}")
    
    markdown_sections.append("")
    
    # Section 7: Pyomo Implementation
    pyomo_result = solver_results.get("pyomo", {})
    pyomo_code = solver_codes.get("pyomo", {}).get("extracted_code", "")
    pyomo_analysis = solver_analysis_result.get("solver_analysis", {}).get("pyomo_analysis", {})
    
    markdown_sections.append("## 7. Pyomo Implementation")
    markdown_sections.append("")
    markdown_sections.append("```python")
    markdown_sections.append(pyomo_code if pyomo_code else "# Code generation failed")
    markdown_sections.append("```")
    markdown_sections.append("")
    
    # Enhanced execution results
    markdown_sections.append("### Execution Results")
    if pyomo_result.get("status") == "optimal":
        markdown_sections.append("**Status**: OPTIMAL")
        optimal_val = pyomo_result.get('optimal_value', 'N/A')
        if optimal_val != 'N/A':
            markdown_sections.append(f"**Optimal Value**: {optimal_val}")
        else:
            markdown_sections.append("**Optimal Value**: N/A")
        exec_time = pyomo_result.get('execution_time', 0)
        markdown_sections.append(f"**Execution Time**: {exec_time:.2f} seconds")
        
        # Decision variables with business meaning
        decision_vars = pyomo_result.get("decision_variables", {})
        if decision_vars:
            markdown_sections.append("**Decision Variables**:")
            for var_name, var_value in decision_vars.items():
                business_meaning = solver_analysis_result.get("business_insights", {}).get("decision_variable_meanings", {}).get(var_name, f"Resource allocation for {var_name}")
                markdown_sections.append(f"- {var_name} = {var_value:.3f} ({business_meaning})")
        
        markdown_sections.append(f"**Reliability**: {pyomo_analysis.get('reliability', 'N/A')}")
        
        # Add retry information if available
        if pyomo_result.get("retry_attempt"):
            markdown_sections.append(f"**Retry Attempt**: {pyomo_result.get('retry_attempt')}")
    else:
        status = pyomo_result.get("status", "unknown").upper()
        markdown_sections.append(f"**Status**: {status}")
        markdown_sections.append(f"**Error**: {pyomo_result.get('error_message', 'Unknown error')}")
    
    if pyomo_analysis.get('notes'):
        markdown_sections.append(f"**Analysis**: {pyomo_analysis.get('notes')}")
    
    markdown_sections.append("")
    
    # Section 8: Cross-Solver Analysis and Final Recommendation
    consistency_eval = solver_analysis_result.get("consistency_evaluation", {})
    final_recommendation = solver_analysis_result.get("final_recommendation", {})
    business_insights = solver_analysis_result.get("business_insights", {})
    
    markdown_sections.append("## 8. Cross-Solver Analysis and Final Recommendation")
    markdown_sections.append("")
    
    # Solver Results Comparison Table
    markdown_sections.append("### Solver Results Comparison")
    markdown_sections.append("")
    markdown_sections.append("| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |")
    markdown_sections.append("|--------|--------|---------------|----------------|-------------------|---------------|")
    
    for solver_name in ["gurobipy", "docplex", "pyomo"]:
        result = solver_results.get(solver_name, {})
        status = result.get("status", "unknown")
        optimal_val = result.get("optimal_value")
        exec_time = result.get("execution_time", 0)
        decision_vars = result.get("decision_variables", {})
        retry_attempt = result.get("retry_attempt", "N/A")
        
        # Format optimal value
        if optimal_val is not None:
            optimal_str = f"{optimal_val:.2f}"
        else:
            optimal_str = "N/A"

        if decision_vars:
            vars_str = ", ".join([f"{k}={v:.2f}" for k, v in list(decision_vars.items())[:3]])
            if len(decision_vars) > 3:
                vars_str += "..."
        else:
            vars_str = "N/A"
        
        # Format retry attempt
        retry_str = str(retry_attempt) if retry_attempt != "N/A" else "N/A"
        
        markdown_sections.append(f"| {solver_name.title()} | {status.upper()} | {optimal_str} | {exec_time:.2f}s | {vars_str} | {retry_str} |")
    
    markdown_sections.append("")
    
    # Consistency Analysis
    markdown_sections.append("### Solver Consistency Analysis")
    values_consistent = consistency_eval.get("values_consistent", False)
    if values_consistent:
        markdown_sections.append("**Result**: All solvers produced consistent results ")
        consistent_solvers = consistency_eval.get("consistent_solvers", [])
        markdown_sections.append(f"**Consistent Solvers**: {', '.join(consistent_solvers)}")
    else:
        markdown_sections.append("**Result**: Solvers produced inconsistent results")
        consistent_solvers = consistency_eval.get("consistent_solvers", [])
        inconsistent_solvers = consistency_eval.get("inconsistent_solvers", [])
        if consistent_solvers:
            markdown_sections.append(f"**Consistent Solvers**: {', '.join(consistent_solvers)}")
        if inconsistent_solvers:
            markdown_sections.append(f"**Inconsistent Solvers**: {', '.join(inconsistent_solvers)}")
        
        potential_issues = consistency_eval.get("potential_issues", [])
        if potential_issues:
            markdown_sections.append("**Potential Issues**:")
            for issue in potential_issues:
                markdown_sections.append(f"- {issue}")
    
    # Majority Vote Result
    major_vote_value = consistency_eval.get("major_vote_optimal_value")
    if major_vote_value is not None:
        markdown_sections.append(f"**Majority Vote Optimal Value**: {major_vote_value}")
    
    # Add retry summary if retries occurred
    retry_summary = []
    for solver_name in ["gurobipy", "docplex", "pyomo"]:
        result = solver_results.get(solver_name, {})
        if result.get("retry_attempt"):
            retry_summary.append(f"{solver_name}: {result.get('retry_attempt')} attempts")
    
    if retry_summary:
        markdown_sections.append(f"**Solver Retry Summary**: {', '.join(retry_summary)}")
    
    markdown_sections.append("")
    # Final Recommendation
    markdown_sections.append("### Final Recommendation")
    recommended_value = final_recommendation.get("recommended_optimal_value")
    confidence = final_recommendation.get("confidence", "medium")
    solver_preference = final_recommendation.get("solver_preference", "multiple")
    
    if recommended_value is not None:
        markdown_sections.append(f"**Recommended Optimal Value**: {recommended_value}")
    markdown_sections.append(f"**Confidence Level**: {confidence.upper()}")
    markdown_sections.append(f"**Preferred Solver(s)**: {solver_preference}")
    
    reasoning = final_recommendation.get("reasoning", "")
    if reasoning:
        markdown_sections.append(f"**Reasoning**: {reasoning}")
    
    # Decision Variables with Business Interpretation
    recommended_vars = final_recommendation.get("recommended_decision_variables", {})
    if recommended_vars:
        markdown_sections.append("")
        markdown_sections.append("### Optimal Decision Variables")
        var_meanings = business_insights.get("decision_variable_meanings", {})
        for var_name, var_value in recommended_vars.items():
            business_meaning = var_meanings.get(var_name, f"Resource allocation for {var_name}")
            markdown_sections.append(f"- **{var_name}** = {var_value:.3f}")
            markdown_sections.append(f"  - *Business Meaning*: {business_meaning}")
    
    markdown_sections.append("")
    
    # Business Insights
    markdown_sections.append("### Business Interpretation")
    
    business_interpretation = final_recommendation.get("business_interpretation", "")
    if business_interpretation:
        markdown_sections.append(f"**Overall Strategy**: {business_interpretation}")
    
    objective_interpretation = business_insights.get("objective_value_interpretation", "")
    if objective_interpretation:
        markdown_sections.append(f"**Objective Value Meaning**: {objective_interpretation}")
    
    resource_summary = business_insights.get("resource_allocation_summary", "")
    if resource_summary:
        markdown_sections.append(f"**Resource Allocation Summary**: {resource_summary}")
    
    implementation_recs = business_insights.get("implementation_recommendations", "")
    if implementation_recs:
        markdown_sections.append(f"**Implementation Recommendations**: {implementation_recs}")
    
    return "\n".join(markdown_sections)
