# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def optimize_architect_assignments():
    """Optimize architect assignments to bridge and mill projects"""

    # 1. MODEL CREATION
    model = pyo.ConcreteModel()

    # 2. DATA SETUP
    # Example data based on the problem context
    bridge_lengths = {1: 150.0, 2: 250.0, 3: 180.0}
    mill_designed = {1: 1, 2: 1, 3: 0}
    architect_capacity = {1: 2, 2: 2, 3: 2}  # Example capacities

    # Validate data lengths
    assert len(bridge_lengths) == len(mill_designed), "Data length mismatch"

    # 3. SETS
    model.I = pyo.Set(initialize=[1, 2, 3])  # Architects
    model.J = pyo.Set(initialize=bridge_lengths.keys())  # Bridges
    model.K = pyo.Set(initialize=mill_designed.keys())  # Mills

    # 4. PARAMETERS
    model.length_meters = pyo.Param(model.J, initialize=bridge_lengths)
    model.designed = pyo.Param(model.K, initialize=mill_designed)
    model.capacity = pyo.Param(model.I, initialize=architect_capacity)

    # 5. VARIABLES
    model.x = pyo.Var(model.I, model.J, within=pyo.Binary)
    model.y = pyo.Var(model.I, model.K, within=pyo.Binary)

    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.length_meters[j] * model.x[i, j] for i in model.I for j in model.J) + \
               sum(model.designed[k] * model.y[i, k] for i in model.I for k in model.K)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)

    # 7. CONSTRAINTS

    # Architect Capacity Constraint
    def capacity_rule(model, i):
        return sum(model.x[i, j] for j in model.J) + sum(model.y[i, k] for k in model.K) <= model.capacity[i]
    model.capacity_constraint = pyo.Constraint(model.I, rule=capacity_rule)

    # Bridge Assignment Constraint
    def bridge_assignment_rule(model, j):
        return sum(model.x[i, j] for i in model.I) == 1
    model.bridge_assignment_constraint = pyo.Constraint(model.J, rule=bridge_assignment_rule)

    # Mill Assignment Constraint
    def mill_assignment_rule(model, k):
        return sum(model.y[i, k] for i in model.I) == 1
    model.mill_assignment_constraint = pyo.Constraint(model.K, rule=mill_assignment_rule)

    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')

    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap

    # Solve the model
    results = solver.solve(model, tee=True)

    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")

        # Extract variable values
        print("\nAssignment values:")
        for i in model.I:
            for j in model.J:
                x_val = pyo.value(model.x[i, j])
                if x_val > 1e-6:
                    print(f"x[{i},{j}] = {int(x_val)}")
            for k in model.K:
                y_val = pyo.value(model.y[i, k])
                if y_val > 1e-6:
                    print(f"y[{i},{k}] = {int(y_val)}")

    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")

    return model

# Run the optimization
optimize_architect_assignments()