# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_staff_allocation():
    """Optimize staff allocation for coffee shop happy hours"""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("coffee_shop")
    
    # Data from the problem context
    hh_ids = [101, 102, 103]
    scores = [4.5, 3.8, 4.2]
    max_staff = [5, 8, 6]
    total_staff_available = 19
    min_staff_per_event = 2
    
    # CRITICAL: Validate array lengths before loops
    assert len(hh_ids) == len(scores) == len(max_staff), "Array length mismatch"
    
    # 2. VARIABLES
    # Decision variables for staff allocation
    x = {i: model.addVar(vtype=GRB.INTEGER, name=f"x_{hh_ids[i]}", lb=0) 
         for i in range(len(hh_ids))}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the overall customer satisfaction score
    model.setObjective(gp.quicksum(scores[i] * x[i] for i in range(len(hh_ids))), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total staff allocation constraint
    model.addConstr(gp.quicksum(x[i] for i in range(len(hh_ids))) <= total_staff_available, name="total_staff_limit")
    
    # Minimum staff requirement for each happy hour event
    for i in range(len(hh_ids)):
        model.addConstr(x[i] >= min_staff_per_event, name=f"min_staff_{hh_ids[i]}")
    
    # Maximum staff available at each shop
    for i in range(len(hh_ids)):
        model.addConstr(x[i] <= max_staff[i], name=f"max_staff_{hh_ids[i]}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(len(hh_ids)):
            if x[i].x > 1e-6:
                print(f"x[{hh_ids[i]}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_staff_allocation()