# Complete Optimization Problem and Solution: county_public_safety

## 1. Problem Context and Goals

### Context  
The objective is to optimize the allocation of police officers across various counties to effectively minimize the overall crime rate. This involves making strategic decisions about the number of police officers assigned to each county. The decision variable in this context is the number of police officers allocated to each county, which is represented as an integer value. The operational parameters are aligned with the goal of minimizing the total crime rate, which is calculated by considering the crime rate in each county and the number of officers assigned there.

The business configuration includes several key parameters: the total number of police officers available for allocation, which serves as a constraint to ensure that the total number of officers assigned does not exceed this limit; the minimum number of officers required in each county, which ensures a baseline level of police presence; and the maximum case burden that can be handled by a single officer, which prevents overloading officers with too many cases. These parameters are crucial for maintaining operational efficiency and ensuring public safety.

The data presented reflects current operational information, focusing on precise decision-making that leads to linear formulations. The constraints are designed to match expected linear relationships, avoiding any scenarios that would require nonlinear relationships such as variable products or divisions. The business configuration parameters are referenced appropriately to ensure a clear understanding of the optimization problem.

### Goals  
The primary goal of this optimization problem is to minimize the total crime rate across all counties. This is achieved by strategically allocating police officers to each county in a way that reduces the overall crime rate. The metric used to measure success is the total crime rate, which is minimized by considering the crime rate in each county and the number of officers assigned there. The optimization goal is clearly defined in natural language, focusing on the linear relationship between the crime rate and the allocation of police officers.

## 2. Constraints    

The optimization problem is subject to several constraints that ensure the solution is feasible and practical:

- The total number of police officers allocated across all counties must not exceed the total number of officers available. This constraint ensures that the allocation does not surpass the resources available.
- Each county must have at least a minimum number of officers assigned to it. This constraint guarantees a baseline level of police presence in every county to maintain public safety.
- The case burden per officer must not exceed the maximum allowed. This constraint prevents any single officer from being overloaded with too many cases, ensuring effective case management and officer well-being.

These constraints are described in business terms that naturally lead to linear mathematical forms, ensuring that the optimization problem remains linear and manageable.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 1 Database Schema
-- Objective: Schema adjustments and configuration logic updates were made to address missing optimization requirements and improve mapping adequacy based on OR expert analysis.

CREATE TABLE county_public_safety (
  Crime_rate FLOAT,
  Population INTEGER,
  Residents_per_officer INTEGER,
  Case_burden FLOAT,
  Police_officers INTEGER
);
```

### Data Dictionary  
The `county_public_safety` table is designed to store data related to public safety optimization across counties. Each column serves a specific purpose in the optimization process:

- **Crime_rate**: Represents the crime rate in each county. It is used to weigh the impact of police officers on crime reduction.
- **Population**: Indicates the population of each county. This data helps determine the maximum number of officers based on the number of residents per officer.
- **Residents_per_officer**: Shows the number of residents per officer in each county. It is used to limit the number of officers allocated.
- **Case_burden**: Reflects the case burden in each county. This ensures that officers are not overloaded with too many cases.
- **Police_officers**: Represents the number of police officers allocated to each county. This is the decision variable for the optimization process.

### Current Stored Values  
```sql
-- Iteration 1 Realistic Data
-- Generated by triple expert (business + data + optimization)
-- Values were generated based on typical crime rates, population sizes, and police allocation strategies observed in medium-sized counties. The approach ensures that the data reflects realistic scenarios for public safety optimization.

-- Realistic data for county_public_safety
INSERT INTO county_public_safety (Crime_rate, Population, Residents_per_officer, Case_burden, Police_officers) VALUES (0.15, 1500, 150, 7.0, 10);
INSERT INTO county_public_safety (Crime_rate, Population, Residents_per_officer, Case_burden, Police_officers) VALUES (0.25, 2500, 125, 12.0, 20);
INSERT INTO county_public_safety (Crime_rate, Population, Residents_per_officer, Case_burden, Police_officers) VALUES (0.18, 1800, 180, 9.0, 12);
INSERT INTO county_public_safety (Crime_rate, Population, Residents_per_officer, Case_burden, Police_officers) VALUES (0.22, 2200, 110, 11.0, 18);
INSERT INTO county_public_safety (Crime_rate, Population, Residents_per_officer, Case_burden, Police_officers) VALUES (0.2, 2000, 100, 10.0, 15);
```

## 4. Mathematical Optimization Formulation

#### Decision Variables
- Let \( x_i \) be the number of police officers allocated to county \( i \).
- \( x_i \) is an integer variable for each county.

#### Objective Function
Minimize the total crime rate across all counties, which can be expressed as:
\[ \text{Minimize } \sum_{i} \text{Crime\_rate}_i \times x_i \]

#### Constraints
1. **Total Officers Constraint**: The total number of police officers allocated across all counties must not exceed the total number of officers available.
   \[
   \sum_{i} x_i \leq \text{Total\_Officers}
   \]
   - Data Source Verification: Total number of officers available is a business configuration parameter.

2. **Minimum Officers per County**: Each county must have at least a minimum number of officers assigned.
   \[
   x_i \geq \text{Minimum\_Officers}_i \quad \forall i
   \]
   - Data Source Verification: Minimum number of officers is a business configuration parameter.

3. **Residents per Officer Constraint**: The number of officers allocated to each county should not exceed the number of residents divided by the residents per officer.
   \[
   x_i \leq \frac{\text{Population}_i}{\text{Residents\_per\_officer}_i} \quad \forall i
   \]
   - Data Source Verification: \(\text{Population}_i\) and \(\text{Residents\_per\_officer}_i\) are from the `county_public_safety` table.

4. **Case Burden Constraint**: The case burden per officer must not exceed the maximum allowed.
   \[
   \frac{\text{Case\_burden}_i}{x_i} \leq \text{Max\_Case\_Burden} \quad \forall i
   \]
   - This constraint can be linearized by ensuring:
   \[
   x_i \geq \frac{\text{Case\_burden}_i}{\text{Max\_Case\_Burden}} \quad \forall i
   \]
   - Data Source Verification: \(\text{Case\_burden}_i\) is from the `county_public_safety` table, and \(\text{Max\_Case\_Burden}\) is a business configuration parameter.

#### Data Source Verification
- \(\text{Crime\_rate}_i\), \(\text{Population}_i\), \(\text{Residents\_per\_officer}_i\), and \(\text{Case\_burden}_i\) are from the `county_public_safety` table.
- \(\text{Total\_Officers}\), \(\text{Minimum\_Officers}_i\), and \(\text{Max\_Case\_Burden}\) are business configuration parameters.

This linear model is immediately solvable using standard linear programming solvers, ensuring that all constraints and the objective function are linear.

## 5. Gurobipy Implementation

```python
# Complete GUROBIPY implementation - Retry Attempt 4

import gurobipy as gp
from gurobipy import GRB

def optimize_police_allocation():
    # 1. MODEL & DATA SETUP
    model = gp.Model("county_public_safety")
    
    # Data from the database
    crime_rates = [0.15, 0.25, 0.18, 0.22, 0.2]
    populations = [1500, 2500, 1800, 2200, 2000]
    residents_per_officer = [150, 125, 180, 110, 100]
    case_burdens = [7.0, 12.0, 9.0, 11.0, 10.0]
    min_officers = [10, 20, 12, 18, 15]
    
    # Business configuration parameters
    total_officers = 75
    max_case_burden = 15.0
    
    # Validate array lengths
    n_counties = len(crime_rates)
    assert len(populations) == len(residents_per_officer) == len(case_burdens) == len(min_officers) == n_counties, "Array length mismatch"
    
    # 2. VARIABLES
    x = model.addVars(n_counties, vtype=GRB.INTEGER, name="x", lb=0)
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(gp.quicksum(crime_rates[i] * x[i] for i in range(n_counties)), GRB.MINIMIZE)
    
    # 4. CONSTRAINTS
    # Total Officers Constraint
    model.addConstr(gp.quicksum(x[i] for i in range(n_counties)) <= total_officers, name="total_officers")
    
    # Minimum Officers per County
    model.addConstrs((x[i] >= min_officers[i] for i in range(n_counties)), name="min_officers")
    
    # Residents per Officer Constraint
    model.addConstrs((x[i] <= populations[i] // residents_per_officer[i] for i in range(n_counties)), name="residents_per_officer")
    
    # Case Burden Constraint
    model.addConstrs((x[i] >= case_burdens[i] / max_case_burden for i in range(n_counties)), name="case_burden")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_counties):
            print(f"x[{i}] = {x[i].x:.0f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_police_allocation()
```

### Execution Results
**Status**: INFEASIBLE
**Error**: Problem is infeasible
**Analysis**: Gurobipy consistently reports the problem as infeasible, indicating a potential issue with the constraints or data.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def optimize_police_allocation():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="county_public_safety")
    
    # Data from the database
    crime_rates = [0.15, 0.25, 0.18, 0.22, 0.2]
    populations = [1500, 2500, 1800, 2200, 2000]
    residents_per_officer = [150, 125, 180, 110, 100]
    case_burdens = [7.0, 12.0, 9.0, 11.0, 10.0]
    min_officers = [10, 20, 12, 18, 15]
    
    # Business configuration parameters
    total_officers = 75
    max_case_burden = 15.0
    
    # Validate array lengths
    n_counties = len(crime_rates)
    assert len(populations) == len(residents_per_officer) == len(case_burdens) == len(min_officers) == n_counties, "Array length mismatch"
    
    safe_range = range(n_counties)  # Safe indexing
    
    # 2. VARIABLES
    x = {i: mdl.integer_var(name=f"x_{i}", lb=0) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(crime_rates[i] * x[i] for i in safe_range)
    mdl.minimize(objective)
    
    # 4. CONSTRAINTS
    # Total Officers Constraint
    mdl.add_constraint(mdl.sum(x[i] for i in safe_range) <= total_officers, ctname="total_officers")
    
    # Minimum Officers per County
    for i in safe_range:
        mdl.add_constraint(x[i] >= min_officers[i], ctname=f"min_officers_{i}")
    
    # Residents per Officer Constraint
    for i in safe_range:
        mdl.add_constraint(x[i] <= populations[i] // residents_per_officer[i], ctname=f"residents_per_officer_{i}")
    
    # Case Burden Constraint
    for i in safe_range:
        mdl.add_constraint(x[i] >= case_burdens[i] / max_case_burden, ctname=f"case_burden_{i}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            if value > 1e-6:
                print(f"x[{i}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

    return mdl

# Execute the optimization
optimize_police_allocation()
```

### Execution Results
**Status**: INFEASIBLE
**Error**: Problem is infeasible
**Analysis**: DOCplex also reports infeasibility, suggesting that the problem formulation or data constraints are not satisfiable.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation - Retry Attempt 4

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def optimize_police_allocation():
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()

    # 2. DATA SETUP
    # Data from the county_public_safety table
    crime_rates = [0.15, 0.25, 0.18, 0.22, 0.2]
    populations = [1500, 2500, 1800, 2200, 2000]
    residents_per_officer = [150, 125, 180, 110, 100]
    case_burdens = [7.0, 12.0, 9.0, 11.0, 10.0]
    min_officers = [10, 20, 12, 18, 15]
    total_officers = 75
    max_case_burden = 10.0

    # Validate array lengths
    n_counties = len(crime_rates)
    assert len(populations) == len(residents_per_officer) == len(case_burdens) == len(min_officers) == n_counties, "Array length mismatch"

    # 3. SETS
    model.I = pyo.RangeSet(1, n_counties)

    # 4. PARAMETERS
    model.crime_rate = pyo.Param(model.I, initialize={i+1: crime_rates[i] for i in range(n_counties)})
    model.population = pyo.Param(model.I, initialize={i+1: populations[i] for i in range(n_counties)})
    model.residents_per_officer = pyo.Param(model.I, initialize={i+1: residents_per_officer[i] for i in range(n_counties)})
    model.case_burden = pyo.Param(model.I, initialize={i+1: case_burdens[i] for i in range(n_counties)})
    model.min_officers = pyo.Param(model.I, initialize={i+1: min_officers[i] for i in range(n_counties)})

    # 5. VARIABLES
    model.x = pyo.Var(model.I, within=pyo.NonNegativeIntegers)

    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.crime_rate[i] * model.x[i] for i in model.I)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.minimize)

    # 7. CONSTRAINTS
    def total_officers_rule(model):
        return sum(model.x[i] for i in model.I) <= total_officers
    model.total_officers_constraint = pyo.Constraint(rule=total_officers_rule)

    def min_officers_rule(model, i):
        return model.x[i] >= model.min_officers[i]
    model.min_officers_constraint = pyo.Constraint(model.I, rule=min_officers_rule)

    def residents_per_officer_rule(model, i):
        return model.x[i] <= model.population[i] / model.residents_per_officer[i]
    model.residents_per_officer_constraint = pyo.Constraint(model.I, rule=residents_per_officer_rule)

    def case_burden_rule(model, i):
        return model.x[i] >= model.case_burden[i] / max_case_burden
    model.case_burden_constraint = pyo.Constraint(model.I, rule=case_burden_rule)

    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    results = solver.solve(model, tee=True)

    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        for i in model.I:
            print(f"x[{i}] = {pyo.value(model.x[i])}")
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")

    return model

# Execute the optimization
optimize_police_allocation()
```

### Execution Results
**Status**: INFEASIBLE
**Error**: Problem is infeasible
**Analysis**: Pyomo's infeasibility result aligns with the other solvers, reinforcing the likelihood of a formulation or data issue.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | INFEASIBLE | N/A | 0.26s | N/A | 4 |
| Docplex | INFEASIBLE | N/A | 7.19s | N/A | 4 |
| Pyomo | INFEASIBLE | N/A | 2.58s | N/A | 4 |

### Solver Consistency Analysis
**Result**: All solvers produced consistent results ✓
**Consistent Solvers**: gurobipy, docplex, pyomo
**Solver Retry Summary**: gurobipy: 4 attempts, docplex: 4 attempts, pyomo: 4 attempts

### Final Recommendation
**Confidence Level**: LOW
**Preferred Solver(s)**: multiple
**Reasoning**: All solvers consistently report infeasibility, indicating a fundamental issue with the problem setup rather than solver performance.

### Business Interpretation
**Overall Strategy**: The current allocation problem cannot be solved with the given constraints and data. This suggests a need to revisit the constraints or data inputs.
**Objective Value Meaning**: The objective value cannot be interpreted as the problem is infeasible. This means the current constraints do not allow for a feasible allocation of officers.
**Resource Allocation Summary**: No feasible allocation of officers can be determined with the current constraints and data.
**Implementation Recommendations**: Review and adjust the constraints and data inputs. Consider relaxing some constraints or verifying the accuracy of the data in the `county_public_safety` table and business configuration parameters.