# Complete DOCPLEX implementation

from docplex.mp.model import Model

def culture_company_optimization():
    """Optimization for Culture Company to maximize cultural impact"""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="culture_company")
    
    # Coefficients for the objective function from Group_Equity_Shareholding
    book_club_coeffs = [0.6, 0.8, 0.5]
    movie_coeffs = [0.7, 0.9, 0.4]
    
    # Budget constraint from Constraint_Bounds
    budget_constraint = 1000000
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(book_club_coeffs) == len(movie_coeffs) == 3, "Array length mismatch"
    safe_range = range(min(len(book_club_coeffs), len(movie_coeffs)))  # Safe indexing
    
    # 2. VARIABLES
    # Investment amounts in book clubs and movies
    x = {i: mdl.continuous_var(name=f"x_{i+1}", lb=0) for i in safe_range}
    y = {i: mdl.continuous_var(name=f"y_{i+1}", lb=0) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total cultural impact
    objective = mdl.sum(book_club_coeffs[i] * x[i] + movie_coeffs[i] * y[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Budget constraint
    total_investment = mdl.sum(x[i] + y[i] for i in safe_range)
    mdl.add_constraint(total_investment <= budget_constraint, ctname="budget_constraint")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            x_value = solution.get_value(x[i])
            y_value = solution.get_value(y[i])
            if x_value > 1e-6:
                print(f"x[{i+1}] = {x_value:.3f}")
            if y_value > 1e-6:
                print(f"y[{i+1}] = {y_value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
culture_company_optimization()