# Complete Optimization Problem and Solution: debate

## 1. Problem Context and Goals

### Context  
A political organization is focused on optimizing the allocation of speakers to various debates. The primary decision involves determining whether a speaker should be assigned to a specific debate. This decision is represented by binary variables, where each variable indicates if a speaker is assigned to a debate. The organization aims to maximize the total audience reached by these debates. The operational parameters include the maximum number of debates a speaker can attend and the maximum number of speakers allowed per debate. These parameters are crucial for ensuring that the allocation remains feasible and effective. The business configuration specifies that each speaker can attend up to a certain number of debates, and each debate can host a limited number of speakers. This setup ensures that the optimization problem remains linear, avoiding complex relationships such as variable products or divisions. The data used in this process reflects current operational realities, ensuring that the optimization aligns with actual business constraints and opportunities.

### Goals  
The primary goal of this optimization is to maximize the total number of audience members reached through the strategic assignment of speakers to debates. The metric for optimization is the sum of audience numbers for each debate, weighted by the assignment of speakers to those debates. Success is measured by the extent to which the total audience is maximized, using the number of audience members as the coefficient source. This goal is articulated in natural language to maintain clarity and focus on the linear nature of the optimization objective.

## 2. Constraints    

The optimization process is governed by two main constraints. First, each speaker is limited to participating in a maximum number of debates, ensuring that their availability is not overstretched. This constraint is defined by the maximum number of debates a speaker can attend. Second, each debate can accommodate only a limited number of speakers, which is determined by the maximum number of speakers allowed per debate. These constraints are expressed in business terms that naturally lead to linear mathematical forms, ensuring that the optimization remains straightforward and manageable.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 1 Database Schema
-- Objective: Schema changes include creating new tables for decision variables and constraint bounds, modifying existing tables to fill mapping gaps, and moving scalar parameters to configuration logic.

CREATE TABLE debate (
  Debate_ID INTEGER,
  Num_of_Audience INTEGER
);

CREATE TABLE decision_variables (
  Debate_ID INTEGER,
  People_ID INTEGER,
  assignment BOOLEAN
);

CREATE TABLE constraint_bounds (
  Constraint_Name STRING,
  Value INTEGER
);
```

### Data Dictionary  
The data dictionary provides a comprehensive overview of the tables and columns used in the optimization process, highlighting their business purposes and roles in the optimization:

- **Debate Table**: This table stores information about each debate, including the number of audience members expected. The audience size serves as a coefficient in the optimization objective, helping to prioritize debates with larger audiences.

  - **Debate_ID**: A unique identifier for each debate, used to index debates in the optimization process.
  - **Num_of_Audience**: Represents the number of audience members for each debate, serving as a key factor in the optimization objective.

- **Decision Variables Table**: This table captures the binary decision variables that indicate whether a speaker is assigned to a debate.

  - **Debate_ID**: Identifies the debate in question, linking the decision variable to a specific debate.
  - **People_ID**: Identifies the speaker, linking the decision variable to a specific individual.
  - **Assignment**: A binary indicator showing whether a speaker is assigned to a debate, forming the core decision variable in the optimization.

- **Constraint Bounds Table**: This table defines the bounds for constraints related to debates and speakers.

  - **Constraint_Name**: Names the constraint, distinguishing between different types of constraints such as speaker participation limits and debate capacity limits.
  - **Value**: Specifies the bound value for each constraint, ensuring that the optimization adheres to business rules and limitations.

### Current Stored Values  
```sql
-- Iteration 1 Realistic Data
-- Generated by triple expert (business + data + optimization)
-- Values were chosen based on typical constraints in political organizations, ensuring a balance between speaker availability and debate audience size.

-- Realistic data for debate
INSERT INTO debate (Debate_ID, Num_of_Audience) VALUES (1, 150);
INSERT INTO debate (Debate_ID, Num_of_Audience) VALUES (2, 250);
INSERT INTO debate (Debate_ID, Num_of_Audience) VALUES (3, 100);

-- Realistic data for decision_variables
INSERT INTO decision_variables (Debate_ID, People_ID, assignment) VALUES (1, 101, True);
INSERT INTO decision_variables (Debate_ID, People_ID, assignment) VALUES (1, 102, False);
INSERT INTO decision_variables (Debate_ID, People_ID, assignment) VALUES (2, 101, True);
INSERT INTO decision_variables (Debate_ID, People_ID, assignment) VALUES (2, 103, True);
INSERT INTO decision_variables (Debate_ID, People_ID, assignment) VALUES (3, 102, True);

-- Realistic data for constraint_bounds
INSERT INTO constraint_bounds (Constraint_Name, Value) VALUES ('Max_Debates_Per_Speaker', 3);
INSERT INTO constraint_bounds (Constraint_Name, Value) VALUES ('Max_Speakers_Per_Debate', 5);
```

## 4. Mathematical Optimization Formulation

#### Decision Variables
Let \( x_{ij} \) be a binary decision variable where:
- \( x_{ij} = 1 \) if speaker \( i \) is assigned to debate \( j \)
- \( x_{ij} = 0 \) otherwise

#### Objective Function
Maximize the total audience reached by the debates:
\[ \text{Maximize } \sum_{j} \text{Num\_of\_Audience}_j \times \sum_{i} x_{ij} \]

Where:
- \(\text{Num\_of\_Audience}_j\) is the number of audience members for debate \( j \) from the `debate.Num_of_Audience` column.

#### Constraints
1. **Speaker Participation Limit**: Each speaker can attend a maximum number of debates.
   \[
   \sum_{j} x_{ij} \leq \text{Max\_Debates\_Per\_Speaker} \quad \forall i
   \]
   Where:
   - \(\text{Max\_Debates\_Per\_Speaker}\) is the maximum number of debates a speaker can attend from `constraint_bounds.Value` where `Constraint_Name` is 'Max_Debates_Per_Speaker'.

2. **Debate Capacity Limit**: Each debate can have a maximum number of speakers.
   \[
   \sum_{i} x_{ij} \leq \text{Max\_Speakers\_Per\_Debate} \quad \forall j
   \]
   Where:
   - \(\text{Max\_Speakers\_Per\_Debate}\) is the maximum number of speakers allowed per debate from `constraint_bounds.Value` where `Constraint_Name` is 'Max_Speakers_Per_Debate'.

3. **Binary Decision Variables**: Each decision variable is binary.
   \[
   x_{ij} \in \{0, 1\} \quad \forall i, j
   \]

Data Source Verification:
- The coefficients for the objective function, \(\text{Num\_of\_Audience}_j\), are sourced from `debate.Num_of_Audience`.
- The constraint bounds, \(\text{Max\_Debates\_Per\_Speaker}\) and \(\text{Max\_Speakers\_Per\_Debate}\), are sourced from `constraint_bounds.Value` for the respective constraints.

## 5. Gurobipy Implementation

```python
# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_speaker_allocation():
    """Optimize the allocation of speakers to debates to maximize audience reach."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("speaker_allocation")
    
    # Data from the database
    debates = [
        {"Debate_ID": 1, "Num_of_Audience": 150},
        {"Debate_ID": 2, "Num_of_Audience": 250},
        {"Debate_ID": 3, "Num_of_Audience": 100}
    ]
    
    speakers = [101, 102, 103]  # Example speaker IDs
    
    # Constraint bounds
    max_debates_per_speaker = 3
    max_speakers_per_debate = 5
    
    # Validate data lengths
    assert len(debates) > 0, "No debates available"
    assert len(speakers) > 0, "No speakers available"
    
    # 2. VARIABLES
    # Create binary decision variables x[i, j] for each speaker i and debate j
    x = model.addVars(
        [(i, j['Debate_ID']) for i in speakers for j in debates],
        vtype=GRB.BINARY,
        name="x"
    )
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total audience reached
    model.setObjective(
        gp.quicksum(j['Num_of_Audience'] * x[i, j['Debate_ID']] for i in speakers for j in debates),
        GRB.MAXIMIZE
    )
    
    # 4. CONSTRAINTS
    # Speaker Participation Limit
    model.addConstrs(
        (gp.quicksum(x[i, j['Debate_ID']] for j in debates) <= max_debates_per_speaker for i in speakers),
        name="speaker_participation_limit"
    )
    
    # Debate Capacity Limit
    model.addConstrs(
        (gp.quicksum(x[i, j['Debate_ID']] for i in speakers) <= max_speakers_per_debate for j in debates),
        name="debate_capacity_limit"
    )
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in speakers:
            for j in debates:
                if x[i, j['Debate_ID']].x > 1e-6:
                    print(f"Speaker {i} assigned to Debate {j['Debate_ID']}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_speaker_allocation()
```

### Execution Results
**Status**: ERROR
**Error**: Traceback (most recent call last):
  File "/tmp/tmpbt_ywjgj.py", line 74, in <module>
    optimize_speaker_allocation()
  File "/tmp/tmpbt_ywjgj.py", line 52, in optimize_speaker_allocation
    model.addConstrs(
  File "src/gurobipy/_model.pyx", line 4095, in gurobipy._model.Model.addConstrs
  File "src/gurobipy/tupledict.pxi", line 144, in gurobipy._core.tupledict.__setitem__
TypeError: unhashable type: 'dict'

**Analysis**: Gurobipy encountered a TypeError due to an unhashable type 'dict'. This indicates a coding error in the constraint setup, preventing the solver from executing properly.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_speaker_allocation():
    """Optimize the allocation of speakers to debates to maximize audience reach."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="speaker_allocation")
    
    # Data from the database
    debates = [
        {"Debate_ID": 1, "Num_of_Audience": 150},
        {"Debate_ID": 2, "Num_of_Audience": 250},
        {"Debate_ID": 3, "Num_of_Audience": 100}
    ]
    
    decision_variables = [
        {"Debate_ID": 1, "People_ID": 101, "assignment": True},
        {"Debate_ID": 1, "People_ID": 102, "assignment": False},
        {"Debate_ID": 2, "People_ID": 101, "assignment": True},
        {"Debate_ID": 2, "People_ID": 103, "assignment": True},
        {"Debate_ID": 3, "People_ID": 102, "assignment": True}
    ]
    
    constraint_bounds = {
        "Max_Debates_Per_Speaker": 3,
        "Max_Speakers_Per_Debate": 5
    }
    
    # Extract unique speaker and debate IDs
    speaker_ids = set(dv["People_ID"] for dv in decision_variables)
    debate_ids = set(d["Debate_ID"] for d in debates)
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(debates) > 0, "No debates available"
    assert len(decision_variables) > 0, "No decision variables available"
    
    # 2. VARIABLES
    x = {(i, j): mdl.binary_var(name=f"x_{i}_{j}") for i in speaker_ids for j in debate_ids}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(debates[j-1]["Num_of_Audience"] * x[i, j] for i in speaker_ids for j in debate_ids)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    
    # Speaker Participation Limit
    for i in speaker_ids:
        mdl.add_constraint(mdl.sum(x[i, j] for j in debate_ids) <= constraint_bounds["Max_Debates_Per_Speaker"], ctname=f"max_debates_speaker_{i}")
    
    # Debate Capacity Limit
    for j in debate_ids:
        mdl.add_constraint(mdl.sum(x[i, j] for i in speaker_ids) <= constraint_bounds["Max_Speakers_Per_Debate"], ctname=f"max_speakers_debate_{j}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in speaker_ids:
            for j in debate_ids:
                value = solution.get_value(x[i, j])
                if value > 1e-6:
                    print(f"x[{i},{j}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
optimize_speaker_allocation()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 1500.0
**Execution Time**: 9.75 seconds
**Reliability**: high
**Analysis**: DOCplex successfully found an optimal solution with an objective value of 1500.0. The execution time was reasonable, indicating a reliable result.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def optimize_speaker_allocation():
    """Optimize speaker allocation to debates to maximize audience reach."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Data from the database schema
    debates = [
        {'Debate_ID': 1, 'Num_of_Audience': 150},
        {'Debate_ID': 2, 'Num_of_Audience': 250},
        {'Debate_ID': 3, 'Num_of_Audience': 100}
    ]
    
    decision_variables = [
        {'Debate_ID': 1, 'People_ID': 101, 'assignment': True},
        {'Debate_ID': 1, 'People_ID': 102, 'assignment': False},
        {'Debate_ID': 2, 'People_ID': 101, 'assignment': True},
        {'Debate_ID': 2, 'People_ID': 103, 'assignment': True},
        {'Debate_ID': 3, 'People_ID': 102, 'assignment': True}
    ]
    
    constraint_bounds = {
        'Max_Debates_Per_Speaker': 3,
        'Max_Speakers_Per_Debate': 5
    }
    
    # Extract unique speaker and debate IDs
    speaker_ids = list(set(dv['People_ID'] for dv in decision_variables))
    debate_ids = list(set(d['Debate_ID'] for d in debates))
    
    # 3. SETS
    model.S = pyo.Set(initialize=speaker_ids)
    model.D = pyo.Set(initialize=debate_ids)
    
    # 4. PARAMETERS
    audience_dict = {d['Debate_ID']: d['Num_of_Audience'] for d in debates}
    model.Num_of_Audience = pyo.Param(model.D, initialize=audience_dict)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.S, model.D, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.Num_of_Audience[j] * model.x[i, j] for i in model.S for j in model.D)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    # Speaker Participation Limit
    def speaker_limit_rule(model, i):
        return sum(model.x[i, j] for j in model.D) <= constraint_bounds['Max_Debates_Per_Speaker']
    model.speaker_limit = pyo.Constraint(model.S, rule=speaker_limit_rule)
    
    # Debate Capacity Limit
    def debate_capacity_rule(model, j):
        return sum(model.x[i, j] for i in model.S) <= constraint_bounds['Max_Speakers_Per_Debate']
    model.debate_capacity = pyo.Constraint(model.D, rule=debate_capacity_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)
    
    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nVariable values:")
        for i in model.S:
            for j in model.D:
                x_val = pyo.value(model.x[i, j])
                if x_val > 1e-6:  # Only print non-zero values
                    print(f"x[{i},{j}] = {int(x_val)}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Run the optimization
optimize_speaker_allocation()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 1500.0
**Execution Time**: 5.69 seconds
**Reliability**: high
**Analysis**: Pyomo also found an optimal solution with the same objective value of 1500.0. The execution time was faster than DOCplex, suggesting efficient performance.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | ERROR | N/A | 0.58s | N/A | N/A |
| Docplex | OPTIMAL | 1500.00 | 9.75s | N/A | N/A |
| Pyomo | OPTIMAL | 1500.00 | 5.69s | N/A | N/A |

### Solver Consistency Analysis
**Result**: All solvers produced consistent results ✓
**Consistent Solvers**: docplex, pyomo
**Majority Vote Optimal Value**: 1500.0

### Final Recommendation
**Recommended Optimal Value**: 1500.0
**Confidence Level**: HIGH
**Preferred Solver(s)**: multiple
**Reasoning**: Both DOCplex and Pyomo provided consistent and optimal results, suggesting that the solution is robust across different solvers.

### Business Interpretation
**Overall Strategy**: The optimal solution maximizes the audience reach to 1500, indicating effective allocation of speakers to debates.
**Objective Value Meaning**: The optimal objective value of 1500 represents the maximum total audience that can be reached by strategically assigning speakers to debates.
**Resource Allocation Summary**: Speakers should be allocated to debates in a way that maximizes audience exposure, adhering to constraints on speaker and debate limits.
**Implementation Recommendations**: Ensure the coding setup for constraints is correct, especially for Gurobipy, and use either DOCplex or Pyomo for reliable results.