# Complete PYOMO implementation - Retry Attempt 4

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def dorm_assignment_optimization():
    """Pyomo implementation for dorm assignment optimization problem"""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    students = [1, 2, 3]
    dorms = [101, 102, 103]
    capacities = {101: 1, 102: 1, 103: 1}
    gender_match = {(1, 101): 1, (2, 102): 1, (3, 103): 1}
    coefficients = {(1, 101): 1.0, (2, 102): 1.0, (3, 103): 0.5}
    
    # CRITICAL: Validate data completeness
    assert all((i, j) in coefficients for i in students for j in dorms), "Coefficient data missing"
    assert all((i, j) in gender_match for i in students for j in dorms), "Gender match data missing"
    
    # 3. SETS
    model.I = pyo.Set(initialize=students)
    model.J = pyo.Set(initialize=dorms)
    
    # 4. PARAMETERS
    model.capacity = pyo.Param(model.J, initialize=capacities)
    model.gender_match = pyo.Param(model.I, model.J, initialize=gender_match)
    model.coefficient = pyo.Param(model.I, model.J, initialize=coefficients)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.I, model.J, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.coefficient[i, j] * model.x[i, j] for i in model.I for j in model.J)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    
    # Capacity constraints
    def capacity_rule(model, j):
        return sum(model.x[i, j] for i in model.I) <= model.capacity[j]
    model.capacity_constraint = pyo.Constraint(model.J, rule=capacity_rule)
    
    # Gender compatibility constraints
    def gender_rule(model, i, j):
        return model.x[i, j] <= model.gender_match[i, j]
    model.gender_constraint = pyo.Constraint(model.I, model.J, rule=gender_rule)
    
    # Assignment constraints
    def assignment_rule(model, i):
        return sum(model.x[i, j] for j in model.J) == 1
    model.assignment_constraint = pyo.Constraint(model.I, rule=assignment_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Solve the model
    results = solver.solve(model, tee=True)
    
    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nAssignment results:")
        for i in model.I:
            for j in model.J:
                if pyo.value(model.x[i, j]) > 0.5:  # Binary variable, check if assigned
                    print(f"Student {i} assigned to Dorm {j}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")

# Execute the optimization
dorm_assignment_optimization()