# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def election_representative_optimization():
    """Optimize campaign resource allocation to maximize seats won."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("election_representative")
    
    # Data from the database (hardcoded for this example)
    seats = [1, 2, 3]
    min_effort = [10.0, 15.0, 20.0]
    max_effort = [50.0, 60.0, 70.0]
    total_campaign_resources = 100.0  # Example total budget
    
    n_representatives = len(seats)
    
    # CRITICAL: Validate array lengths before loops
    assert len(seats) == len(min_effort) == len(max_effort) == n_representatives, "Array length mismatch"
    
    # 2. VARIABLES
    # Variable dictionaries for efforts
    effort = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"effort_{i}", lb=min_effort[i], ub=max_effort[i]) 
              for i in range(n_representatives)}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total number of seats won
    model.setObjective(gp.quicksum(seats[i] * effort[i] for i in range(n_representatives)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total effort constraint
    model.addConstr(gp.quicksum(effort[i] for i in range(n_representatives)) <= total_campaign_resources, name="total_effort_limit")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_representatives):
            if effort[i].x > 1e-6:
                print(f"Effort for representative {i}: {effort[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
election_representative_optimization()