# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def election_representative_optimization():
    """Optimize campaign resource allocation to maximize seats won."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Data from the database
    seats = [1, 2, 3]
    min_effort = [10.0, 15.0, 20.0]
    max_effort = [50.0, 60.0, 70.0]
    total_campaign_resources = 100.0  # Example total budget
    
    # CRITICAL: Validate array lengths before indexing
    n_representatives = len(seats)
    assert len(min_effort) == len(max_effort) == n_representatives, "Array length mismatch"
    
    # 3. SETS
    model.I = pyo.RangeSet(1, n_representatives)  # 1-based indexing
    
    # 4. PARAMETERS
    model.seats = pyo.Param(model.I, initialize={i+1: seats[i] for i in range(n_representatives)})
    model.min_effort = pyo.Param(model.I, initialize={i+1: min_effort[i] for i in range(n_representatives)})
    model.max_effort = pyo.Param(model.I, initialize={i+1: max_effort[i] for i in range(n_representatives)})
    
    # 5. VARIABLES
    model.x = pyo.Var(model.I, within=pyo.NonNegativeReals)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.seats[i] * model.x[i] for i in model.I)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    
    # Total effort constraint
    def total_effort_rule(model):
        return sum(model.x[i] for i in model.I) <= total_campaign_resources
    model.total_effort_constraint = pyo.Constraint(rule=total_effort_rule)
    
    # Minimum and maximum effort constraints
    def effort_bounds_rule(model, i):
        return (model.min_effort[i], model.x[i], model.max_effort[i])
    model.effort_bounds_constraint = pyo.Constraint(model.I, rule=effort_bounds_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nVariable values:")
        for i in model.I:
            x_val = pyo.value(model.x[i])
            if x_val > 1e-6:  # Only print non-zero values
                print(f"x[{i}] = {x_val:.3f}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Run the optimization
election_representative_optimization()