# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def employee_hire_evaluation():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="employee_hire_evaluation")
    
    # Data from the database
    employee_ids = [1, 2, 3]
    bonuses = [500.0, 750.0, 1000.0]
    shop_ids = [101, 102, 103]
    min_employees = [3, 4, 5]
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(employee_ids) == len(bonuses), "Employee and bonus array length mismatch"
    assert len(shop_ids) == len(min_employees), "Shop and min_employees array length mismatch"
    
    # Safe indexing
    safe_employee_range = range(len(employee_ids))
    safe_shop_range = range(len(shop_ids))
    
    # 2. VARIABLES
    # Binary decision variables for employee-shop assignments
    x = {(i, j): mdl.binary_var(name=f"x_{i}_{j}") for i in safe_employee_range for j in safe_shop_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total bonuses
    objective = mdl.sum(bonuses[i] * x[i, j] for i in safe_employee_range for j in safe_shop_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    # Each employee is assigned to exactly one shop
    for i in safe_employee_range:
        mdl.add_constraint(mdl.sum(x[i, j] for j in safe_shop_range) == 1, ctname=f"assign_employee_{i}")
    
    # Each shop meets its minimum staffing requirement
    for j in safe_shop_range:
        mdl.add_constraint(mdl.sum(x[i, j] for i in safe_employee_range) >= min_employees[j], ctname=f"min_staff_{j}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_employee_range:
            for j in safe_shop_range:
                if solution.get_value(x[i, j]) > 0.5:  # Check if the employee is assigned to the shop
                    print(f"Employee {employee_ids[i]} assigned to Shop {shop_ids[j]}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

    return mdl

# Execute the function
employee_hire_evaluation()