# Complete DOCPLEX implementation

from docplex.mp.model import Model

def entertainment_awards_optimization():
    """Optimize artwork nominations for maximum audience impact."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="entertainment_awards")
    
    # Data from the database
    num_of_audience = [1200, 4500, 9000]  # Audience sizes for each festival
    max_nominations = [3, 5, 7]           # Max nominations per artwork
    max_capacity = [50, 100, 150]         # Max capacity per festival
    
    # Number of artworks and festivals
    num_artworks = len(max_nominations)
    num_festivals = len(num_of_audience)
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(max_nominations) == num_artworks, "Array length mismatch for max_nominations"
    assert len(max_capacity) == num_festivals, "Array length mismatch for max_capacity"
    assert len(num_of_audience) == num_festivals, "Array length mismatch for num_of_audience"
    
    # Safe range for indexing
    safe_artworks = range(num_artworks)
    safe_festivals = range(num_festivals)
    
    # 2. VARIABLES
    # Binary decision variables for nominations
    x = {(i, j): mdl.binary_var(name=f"x_{i}_{j}") for i in safe_artworks for j in safe_festivals}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total audience impact of successful nominations
    objective = mdl.sum(num_of_audience[j] * x[i, j] for i in safe_artworks for j in safe_festivals)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    
    # Maximum nominations per artwork
    for i in safe_artworks:
        mdl.add_constraint(mdl.sum(x[i, j] for j in safe_festivals) <= max_nominations[i], ctname=f"max_nominations_artwork_{i}")
    
    # Maximum capacity per festival
    for j in safe_festivals:
        mdl.add_constraint(mdl.sum(x[i, j] for i in safe_artworks) <= max_capacity[j], ctname=f"max_capacity_festival_{j}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_artworks:
            for j in safe_festivals:
                value = solution.get_value(x[i, j])
                if value > 1e-6:
                    print(f"x[{i},{j}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
entertainment_awards_optimization()