# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def flight_company_optimization():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="flight_company")
    
    # Data from the problem statement
    flight_ids = [1, 2, 3]
    airport_ids = [101, 102, 103]
    distances = {
        (1, 101): 550.0,
        (2, 102): 800.0,
        (3, 103): 950.0
    }
    fee_rates = {
        101: 12.0,
        102: 18.0,
        103: 22.0
    }
    capacities = {
        101: 120,
        102: 180,
        103: 150
    }
    
    # CRITICAL: Validate data consistency
    assert len(flight_ids) * len(airport_ids) == len(distances), "Data length mismatch"
    
    # 2. VARIABLES
    x = {(i, j): mdl.binary_var(name=f"x_{i}_{j}") for i in flight_ids for j in airport_ids}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(distances[i, j] * fee_rates[j] * x[i, j] for i in flight_ids for j in airport_ids)
    mdl.minimize(objective)
    
    # 4. CONSTRAINTS
    # Each flight must be assigned to exactly one airport
    for i in flight_ids:
        mdl.add_constraint(mdl.sum(x[i, j] for j in airport_ids) == 1, ctname=f"assign_flight_{i}")
    
    # The total number of flights assigned to an airport must not exceed its capacity
    for j in airport_ids:
        mdl.add_constraint(mdl.sum(x[i, j] for i in flight_ids) <= capacities[j], ctname=f"capacity_airport_{j}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in flight_ids:
            for j in airport_ids:
                if solution.get_value(x[i, j]) > 0.5:  # Binary variable, check if assigned
                    print(f"Flight {i} assigned to Airport {j}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

# Run the optimization
flight_company_optimization()