# Complete DOCPLEX implementation - Retry Attempt 1

from docplex.mp.model import Model

def mountain_photos_optimization():
    """Optimize the selection of mountain photos for a photography book."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="mountain_photos")
    
    # Data: Photo IDs and their diversity scores
    photo_ids = [1, 2, 3, 4, 5]
    diversity_scores = [0.85, 0.75, 0.9, 0.65, 0.8]
    
    # Constraints: Total number of photos and max photos per mountain
    P = 3  # Total number of photos allowed
    M_max = [2, 2, 1, 1, 1]  # Max photos per mountain (example values)
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(photo_ids) == len(diversity_scores) == len(M_max), "Array length mismatch"
    safe_range = range(len(photo_ids))  # Safe indexing
    
    # 2. VARIABLES
    # Binary decision variables for each photo
    x = {i: mdl.binary_var(name=f"x_{photo_ids[i]}") for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total diversity score of the selected photos
    objective = mdl.sum(diversity_scores[i] * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    # Total number of photos constraint
    mdl.add_constraint(mdl.sum(x[i] for i in safe_range) <= P, ctname="total_photos_limit")
    
    # Photos per mountain constraint
    for i in safe_range:
        mdl.add_constraint(x[i] <= M_max[i], ctname=f"mountain_limit_{photo_ids[i]}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            if value > 1e-6:
                print(f"Photo {photo_ids[i]} selected with diversity score {diversity_scores[i]}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

    return mdl

# Execute the optimization
mountain_photos_optimization()