# Complete GUROBIPY implementation - Retry Attempt 1

import gurobipy as gp
from gurobipy import GRB

def optimize_mountain_photos():
    # 1. MODEL & DATA SETUP
    model = gp.Model("mountain_photos")
    
    # Data: Photo IDs and their diversity scores
    photo_ids = [1, 2, 3, 4, 5]
    diversity_scores = [0.85, 0.75, 0.9, 0.65, 0.8]
    
    # Constraints: Total number of photos and max photos per mountain
    P = 3  # Total number of photos allowed
    M_max = [2, 2, 1, 1, 1]  # Max photos per mountain (example values)
    
    # CRITICAL: Validate array lengths before loops
    assert len(photo_ids) == len(diversity_scores) == len(M_max), "Array length mismatch"
    
    # 2. VARIABLES
    # Binary decision variables for each photo
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{i}") for i in photo_ids}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total diversity score of the selected photos
    model.setObjective(gp.quicksum(diversity_scores[i-1] * x[i] for i in photo_ids), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total number of photos constraint
    model.addConstr(gp.quicksum(x[i] for i in photo_ids) <= P, name="total_photos_limit")
    
    # Photos per mountain constraint
    # Assuming each photo belongs to a unique mountain for simplicity
    for i in photo_ids:
        model.addConstr(x[i] <= M_max[i-1], name=f"mountain_limit_{i}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in photo_ids:
            if x[i].x > 1e-6:
                print(f"x[{i}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_mountain_photos()