# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_museum_staff_allocation(total_available_staff):
    """Optimize staff allocation across museums to maximize visitor satisfaction."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="museum_visit")
    
    # Data from the database
    satisfaction_coefficients = [1.3, 1.6, 1.2]
    cost_coefficients = [0.9, 1.1, 0.8]
    minimum_staff_required = [5, 7, 4]
    maximum_staff_capacity = [15, 20, 10]
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(satisfaction_coefficients) == len(cost_coefficients) == len(minimum_staff_required) == len(maximum_staff_capacity) == 3, "Array length mismatch"
    safe_range = range(3)  # Safe indexing for all arrays
    
    # 2. VARIABLES
    # Integer variables for staff allocation
    x = {i: mdl.integer_var(name=f"x_{i}", lb=minimum_staff_required[i], ub=maximum_staff_capacity[i]) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total visitor satisfaction minus total operational cost
    objective = mdl.sum((satisfaction_coefficients[i] - cost_coefficients[i]) * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    # Total staff constraint
    total_staff_usage = mdl.sum(x[i] for i in safe_range)
    mdl.add_constraint(total_staff_usage <= total_available_staff, ctname="total_staff_limit")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            print(f"x[{i}] = {value:.0f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Example usage
total_available_staff = 30  # Example total available staff
optimize_museum_staff_allocation(total_available_staff)