# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_music_storage():
    """Optimize music storage to maximize total rating under constraints."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("music_1")
    
    # Data from the problem statement
    ratings = [4.5, 3.8, 5.0]
    file_sizes = [5000, 3000, 7000]
    resolutions = [1080, 720, 480]
    
    # Assumed parameters
    total_storage_capacity = 10000
    minimum_resolution_quality = 1500
    
    n_songs = len(ratings)
    
    # CRITICAL: Validate array lengths before loops
    assert len(file_sizes) == len(resolutions) == n_songs, "Array length mismatch"
    
    # 2. VARIABLES
    # Binary decision variables for each song
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{i}") for i in range(n_songs)}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total rating of the stored songs
    model.setObjective(gp.quicksum(ratings[i] * x[i] for i in range(n_songs)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Storage Capacity Constraint
    model.addConstr(gp.quicksum(file_sizes[i] * x[i] for i in range(n_songs)) <= total_storage_capacity, name="storage_capacity")
    
    # Resolution Quality Constraint
    model.addConstr(gp.quicksum(resolutions[i] * x[i] for i in range(n_songs)) >= minimum_resolution_quality, name="resolution_quality")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_songs):
            if x[i].x > 1e-6:
                print(f"x[{i}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_music_storage()