## 4. Mathematical Optimization Formulation

#### Decision Variables
- Let \( x_i \) be a binary decision variable for each artist \( i \), where \( x_i = 1 \) if artist \( i \) is selected to perform at the festival, and \( x_i = 0 \) otherwise.

#### Objective Function
- Maximize the total weeks on top of the charts for the selected artists:
  \[
  \text{Maximize } \sum_{i} (\text{Weeks\_on\_Top}_i \times x_i)
  \]
  where \(\text{Weeks\_on\_Top}_i\) is the number of weeks artist \( i \) has been on top of the charts.

#### Constraints
1. **Capacity Constraint**: The total number of selected artists must not exceed the festival's capacity.
   \[
   \sum_{i} x_i \leq \text{Capacity}
   \]

2. **Availability Constraint**: An artist can only be selected if they are available on the festival date.
   \[
   x_i \leq \text{Availability\_Status}_i \quad \forall i
   \]

Data Source Verification:
- \(\text{Weeks\_on\_Top}_i\) comes from the `artist_weeks_on_top.Weeks_on_Top` column.
- \(\text{Capacity}\) comes from the `festival_capacity.Capacity` column.
- \(\text{Availability\_Status}_i\) comes from the `artist.Availability_Status` column.

Using the provided data, the specific formulation becomes:

- Decision Variables:
  - \( x_1, x_2, x_3 \) (binary variables for artists 1, 2, and 3)

- Objective Function:
  \[
  \text{Maximize } 12x_1 + 8x_2 + 15x_3
  \]

- Constraints:
  1. Capacity Constraint:
     \[
     x_1 + x_2 + x_3 \leq 2
     \]

  2. Availability Constraints:
     \[
     x_1 \leq 1
     \]
     \[
     x_2 \leq 0
     \]
     \[
     x_3 \leq 1
     \]

This model can be solved using a linear programming solver to determine the optimal selection of artists that maximizes the festival's exposure while adhering to the constraints.

## 5. Gurobipy Implementation

```python
# Complete GUROBIPY implementation - Retry Attempt 1

import gurobipy as gp
from gurobipy import GRB

def music_festival_optimization():
    """Optimize artist selection for maximum exposure at a music festival."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("music_festival")
    
    # Data from the problem statement
    artist_ids = [1, 2, 3]
    weeks_on_top = [12, 8, 15]
    availability_status = [True, False, True]
    capacity = 2
    
    # Validate array lengths
    assert len(artist_ids) == len(weeks_on_top) == len(availability_status), "Array length mismatch"
    
    # 2. VARIABLES
    # Binary decision variables for each artist
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{artist_ids[i]}") for i in range(len(artist_ids))}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total weeks on top for selected artists
    model.setObjective(gp.quicksum(weeks_on_top[i] * x[i] for i in range(len(artist_ids))), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    # Capacity constraint
    model.addConstr(gp.quicksum(x[i] for i in range(len(artist_ids))) <= capacity, name="capacity_constraint")
    
    # Availability constraints
    for i in range(len(artist_ids)):
        model.addConstr(x[i] <= int(availability_status[i]), name=f"availability_{artist_ids[i]}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(len(artist_ids)):
            if x[i].x > 1e-6:
                print(f"Artist {artist_ids[i]} selected: x[{i}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    else:
        print("Optimization ended with status", model.status)

# Run the optimization
music_festival_optimization()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 27.0
**Execution Time**: 0.21 seconds
**Reliability**: high
**Retry Attempt**: 1
**Analysis**: Gurobipy found an optimal solution quickly with a clear decision variable assignment.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation - Retry Attempt 1

from docplex.mp.model import Model

def music_festival_optimization():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="music_festival")
    
    # Data from the problem statement
    artist_ids = [1, 2, 3]
    weeks_on_top = [12, 8, 15]
    availability_status = [True, False, True]
    capacity = 2
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(artist_ids) == len(weeks_on_top) == len(availability_status), "Array length mismatch"
    safe_range = range(len(artist_ids))  # Safe indexing
    
    # 2. VARIABLES
    x = {i: mdl.binary_var(name=f"x_{artist_ids[i]}") for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(weeks_on_top[i] * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Capacity Constraint
    mdl.add_constraint(mdl.sum(x[i] for i in safe_range) <= capacity, ctname="capacity_constraint")
    
    # Availability Constraints
    for i in safe_range:
        # Convert boolean to integer for constraint compatibility
        availability = 1 if availability_status[i] else 0
        mdl.add_constraint(x[i] <= availability, ctname=f"availability_{artist_ids[i]}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            if value > 0.5:  # Since x[i] is binary, check if selected
                print(f"Artist {artist_ids[i]} is selected.")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

# Run the optimization
music_festival_optimization()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 27.0
**Execution Time**: 8.37 seconds
**Reliability**: high
**Retry Attempt**: 1
**Analysis**: DOCplex also found the optimal solution, albeit with a longer execution time compared to Gurobipy.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation - Retry Attempt 1

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def music_festival_optimization():
    """Optimize artist selection for maximum exposure at a music festival."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Data from the problem statement
    artist_ids = [1, 2, 3]
    weeks_on_top = {1: 12, 2: 8, 3: 15}
    availability_status = {1: True, 2: False, 3: True}
    capacity = 2
    
    # 3. SETS
    model.I = pyo.Set(initialize=artist_ids)
    
    # 4. PARAMETERS
    model.weeks_on_top = pyo.Param(model.I, initialize=weeks_on_top)
    model.availability_status = pyo.Param(model.I, initialize=availability_status)
    model.capacity = pyo.Param(initialize=capacity)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.I, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.weeks_on_top[i] * model.x[i] for i in model.I)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    # Capacity constraint
    def capacity_rule(model):
        return sum(model.x[i] for i in model.I) <= model.capacity
    model.capacity_constraint = pyo.Constraint(rule=capacity_rule)
    
    # Availability constraints
    def availability_rule(model, i):
        return model.x[i] <= int(model.availability_status[i])
    model.availability_constraint = pyo.Constraint(model.I, rule=availability_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Solve the model
    results = solver.solve(model, tee=True)
    
    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nSelected artists:")
        for i in model.I:
            if pyo.value(model.x[i]) > 0.5:  # Binary variable, check if selected
                print(f"Artist {i} selected")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")

# Execute the optimization function
music_festival_optimization()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 27.0
**Execution Time**: 1.16 seconds
**Reliability**: high
**Retry Attempt**: 1
**Analysis**: Pyomo achieved the optimal solution with a moderate execution time, confirming the results of the other solvers.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | OPTIMAL | 27.00 | 0.21s | N/A | 1 |
| Docplex | OPTIMAL | 27.00 | 8.37s | N/A | 1 |
| Pyomo | OPTIMAL | 27.00 | 1.16s | N/A | 1 |

### Solver Consistency Analysis
**Result**: All solvers produced consistent results ✓
**Consistent Solvers**: gurobipy, docplex, pyomo
**Majority Vote Optimal Value**: 27.0
**Solver Retry Summary**: gurobipy: 1 attempts, docplex: 1 attempts, pyomo: 1 attempts

### Final Recommendation
**Recommended Optimal Value**: 27.0
**Confidence Level**: HIGH
**Preferred Solver(s)**: gurobipy
**Reasoning**: Gurobipy provided the optimal solution with the fastest execution time, making it the most efficient choice.

### Optimal Decision Variables
- **x_1** = 1.000
  - *Business Meaning*: Artist 2 is not selected due to unavailability.
- **x_2** = 0.000
  - *Business Meaning*: Artist 3 is selected to perform, contributing 15 weeks on top of the charts.
- **x_3** = 1.000
  - *Business Meaning*: Resource allocation for x_3

### Business Interpretation
**Overall Strategy**: Selecting artists 1 and 3 maximizes the festival's exposure with a total of 27 weeks on top of the charts.
**Objective Value Meaning**: The optimal objective value of 27 indicates the maximum total weeks on top of the charts for the selected artists.
**Resource Allocation Summary**: Allocate resources to secure performances from artists 1 and 3, ensuring maximum exposure.
**Implementation Recommendations**: Confirm the availability and contractual agreements with artists 1 and 3, and proceed with marketing and logistical arrangements for their performances.