# Complete Optimization Problem and Solution: music_4

## 1. Problem Context and Goals

### Context  
The music festival organizer is tasked with selecting a set of artists to perform at an upcoming festival. The primary decision involves choosing which artists to invite, where each artist's selection is represented by a binary decision variable. The goal is to maximize the festival's exposure by selecting artists who have spent the most weeks at the top of the music charts. This decision-making process is constrained by the festival's capacity, which limits the number of artists that can be invited, and the availability of each artist on the festival date.

The operational parameters are straightforward: the objective is to maximize the total number of weeks the selected artists have been on top of the charts. This is achieved by summing the product of each artist's weeks on top and their selection status. The festival's capacity acts as a constraint, ensuring that the number of selected artists does not exceed the venue's limit. Additionally, only artists who are available on the festival date can be selected.

### Goals  
The optimization goal is to maximize the festival's exposure by selecting artists who have the highest cumulative weeks on top of the charts. The metric for optimization is the total weeks on top, calculated by summing the weeks on top for each selected artist. Success is measured by the total exposure achieved, aligning with the objective of maximizing the festival's impact through artist selection.

## 2. Constraints    

The selection of artists is subject to two primary constraints. First, the total number of artists selected must not exceed the festival's capacity. This ensures that the festival remains within logistical and operational limits. Second, an artist can only be selected if they are available on the festival date. These constraints are designed to ensure that the selection process is both feasible and aligned with the festival's operational requirements.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 2 Database Schema
-- Objective: Added Weeks_on_Top table to address missing data for objective coefficients, updated data dictionary, and adjusted business configuration logic for scalar parameters.

CREATE TABLE artist (
  Artist_ID INTEGER,
  Availability_Status BOOLEAN
);

CREATE TABLE artist_weeks_on_top (
  Artist_ID INTEGER,
  Weeks_on_Top INTEGER
);

CREATE TABLE festival_capacity (
  Capacity INTEGER
);
```

### Data Dictionary  
- **Artist Table**: This table contains information about each artist, including a unique identifier and their availability status. The availability status indicates whether an artist is available to perform on the festival date, which is crucial for ensuring that only available artists are considered for selection.

- **Artist Weeks on Top Table**: This table records the number of weeks each artist has been at the top of the music charts. This data serves as the basis for the optimization objective, as it reflects the potential exposure each artist can bring to the festival.

- **Festival Capacity Table**: This table defines the maximum number of artists that can be selected for the festival. It serves as a constraint in the optimization model, ensuring that the number of selected artists does not exceed the festival's logistical capacity.

### Current Stored Values  
```sql
-- Iteration 2 Realistic Data
-- Generated by triple expert (business + data + optimization)
-- Values were determined based on typical festival capacities, artist popularity metrics, and availability patterns observed in the music industry.

-- Realistic data for artist
INSERT INTO artist (Artist_ID, Availability_Status) VALUES (1, True);
INSERT INTO artist (Artist_ID, Availability_Status) VALUES (2, False);
INSERT INTO artist (Artist_ID, Availability_Status) VALUES (3, True);

-- Realistic data for artist_weeks_on_top
INSERT INTO artist_weeks_on_top (Artist_ID, Weeks_on_Top) VALUES (1, 12);
INSERT INTO artist_weeks_on_top (Artist_ID, Weeks_on_Top) VALUES (2, 8);
INSERT INTO artist_weeks_on_top (Artist_ID, Weeks_on_Top) VALUES (3, 15);

-- Realistic data for festival_capacity
INSERT INTO festival_capacity (Capacity) VALUES (2);
```

## 4. Mathematical Optimization Formulation

#### Decision Variables
- Let \( x_i \) be a binary decision variable for each artist \( i \), where \( x_i = 1 \) if artist \( i \) is selected to perform at the festival, and \( x_i = 0 \) otherwise.

#### Objective Function
- Maximize the total weeks on top of the charts for the selected artists:
  \[
  \text{Maximize } \sum_{i} (\text{Weeks\_on\_Top}_i \times x_i)
  \]
  where \(\text{Weeks\_on\_Top}_i\) is the number of weeks artist \( i \) has been on top of the charts.

#### Constraints
1. **Capacity Constraint**: The total number of selected artists must not exceed the festival's capacity.
   \[
   \sum_{i} x_i \leq \text{Capacity}
   \]

2. **Availability Constraint**: An artist can only be selected if they are available on the festival date.
   \[
   x_i \leq \text{Availability\_Status}_i \quad \forall i
   \]

Data Source Verification:
- \(\text{Weeks\_on\_Top}_i\) comes from the `artist_weeks_on_top.Weeks_on_Top` column.
- \(\text{Capacity}\) comes from the `festival_capacity.Capacity` column.
- \(\text{Availability\_Status}_i\) comes from the `artist.Availability_Status` column.

Using the provided data, the specific formulation becomes:

- Decision Variables:
  - \( x_1, x_2, x_3 \) (binary variables for artists 1, 2, and 3)

- Objective Function:
  \[
  \text{Maximize } 12x_1 + 8x_2 + 15x_3
  \]

- Constraints:
  1. Capacity Constraint:
     \[
     x_1 + x_2 + x_3 \leq 2
     \]

  2. Availability Constraints:
     \[
     x_1 \leq 1
     \]
     \[
     x_2 \leq 0
     \]
     \[
     x_3 \leq 1
     \]

This model can be solved using a linear programming solver to determine the optimal selection of artists that maximizes the festival's exposure while adhering to the constraints.

## 5. Gurobipy Implementation

```python
# Complete GUROBIPY implementation - Retry Attempt 1

import gurobipy as gp
from gurobipy import GRB

def music_festival_optimization():
    """Optimize artist selection for maximum exposure at a music festival."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("music_festival")
    
    # Data from the problem statement
    artist_ids = [1, 2, 3]
    weeks_on_top = [12, 8, 15]
    availability_status = [True, False, True]
    capacity = 2
    
    # Validate array lengths
    assert len(artist_ids) == len(weeks_on_top) == len(availability_status), "Array length mismatch"
    
    # 2. VARIABLES
    # Binary decision variables for each artist
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{artist_ids[i]}") for i in range(len(artist_ids))}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total weeks on top for selected artists
    model.setObjective(gp.quicksum(weeks_on_top[i] * x[i] for i in range(len(artist_ids))), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    # Capacity constraint
    model.addConstr(gp.quicksum(x[i] for i in range(len(artist_ids))) <= capacity, name="capacity_constraint")
    
    # Availability constraints
    for i in range(len(artist_ids)):
        model.addConstr(x[i] <= int(availability_status[i]), name=f"availability_{artist_ids[i]}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(len(artist_ids)):
            if x[i].x > 1e-6:
                print(f"Artist {artist_ids[i]} selected: x[{i}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    else:
        print("Optimization ended with status", model.status)

# Run the optimization
music_festival_optimization()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 27.0
**Execution Time**: 0.21 seconds
**Reliability**: high
**Retry Attempt**: 1
**Analysis**: Gurobipy found an optimal solution quickly with a clear decision variable assignment.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation - Retry Attempt 1

from docplex.mp.model import Model

def music_festival_optimization():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="music_festival")
    
    # Data from the problem statement
    artist_ids = [1, 2, 3]
    weeks_on_top = [12, 8, 15]
    availability_status = [True, False, True]
    capacity = 2
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(artist_ids) == len(weeks_on_top) == len(availability_status), "Array length mismatch"
    safe_range = range(len(artist_ids))  # Safe indexing
    
    # 2. VARIABLES
    x = {i: mdl.binary_var(name=f"x_{artist_ids[i]}") for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(weeks_on_top[i] * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Capacity Constraint
    mdl.add_constraint(mdl.sum(x[i] for i in safe_range) <= capacity, ctname="capacity_constraint")
    
    # Availability Constraints
    for i in safe_range:
        # Convert boolean to integer for constraint compatibility
        availability = 1 if availability_status[i] else 0
        mdl.add_constraint(x[i] <= availability, ctname=f"availability_{artist_ids[i]}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            if value > 0.5:  # Since x[i] is binary, check if selected
                print(f"Artist {artist_ids[i]} is selected.")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

# Run the optimization
music_festival_optimization()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 27.0
**Execution Time**: 8.37 seconds
**Reliability**: high
**Retry Attempt**: 1
**Analysis**: DOCplex also found the optimal solution, albeit with a longer execution time compared to Gurobipy.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation - Retry Attempt 1

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def music_festival_optimization():
    """Optimize artist selection for maximum exposure at a music festival."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Data from the problem statement
    artist_ids = [1, 2, 3]
    weeks_on_top = {1: 12, 2: 8, 3: 15}
    availability_status = {1: True, 2: False, 3: True}
    capacity = 2
    
    # 3. SETS
    model.I = pyo.Set(initialize=artist_ids)
    
    # 4. PARAMETERS
    model.weeks_on_top = pyo.Param(model.I, initialize=weeks_on_top)
    model.availability_status = pyo.Param(model.I, initialize=availability_status)
    model.capacity = pyo.Param(initialize=capacity)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.I, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.weeks_on_top[i] * model.x[i] for i in model.I)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    # Capacity constraint
    def capacity_rule(model):
        return sum(model.x[i] for i in model.I) <= model.capacity
    model.capacity_constraint = pyo.Constraint(rule=capacity_rule)
    
    # Availability constraints
    def availability_rule(model, i):
        return model.x[i] <= int(model.availability_status[i])
    model.availability_constraint = pyo.Constraint(model.I, rule=availability_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Solve the model
    results = solver.solve(model, tee=True)
    
    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nSelected artists:")
        for i in model.I:
            if pyo.value(model.x[i]) > 0.5:  # Binary variable, check if selected
                print(f"Artist {i} selected")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")

# Execute the optimization function
music_festival_optimization()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 27.0
**Execution Time**: 1.16 seconds
**Reliability**: high
**Retry Attempt**: 1
**Analysis**: Pyomo achieved the optimal solution with a moderate execution time, confirming the results of the other solvers.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | OPTIMAL | 27.00 | 0.21s | N/A | 1 |
| Docplex | OPTIMAL | 27.00 | 8.37s | N/A | 1 |
| Pyomo | OPTIMAL | 27.00 | 1.16s | N/A | 1 |

### Solver Consistency Analysis
**Result**: All solvers produced consistent results ✓
**Consistent Solvers**: gurobipy, docplex, pyomo
**Majority Vote Optimal Value**: 27.0
**Solver Retry Summary**: gurobipy: 1 attempts, docplex: 1 attempts, pyomo: 1 attempts

### Final Recommendation
**Recommended Optimal Value**: 27.0
**Confidence Level**: HIGH
**Preferred Solver(s)**: gurobipy
**Reasoning**: Gurobipy provided the optimal solution with the fastest execution time, making it the most efficient choice.

### Optimal Decision Variables
- **x_1** = 1.000
  - *Business Meaning*: Artist 2 is not selected due to unavailability.
- **x_2** = 0.000
  - *Business Meaning*: Artist 3 is selected to perform, contributing 15 weeks on top of the charts.
- **x_3** = 1.000
  - *Business Meaning*: Resource allocation for x_3

### Business Interpretation
**Overall Strategy**: Selecting artists 1 and 3 maximizes the festival's exposure with a total of 27 weeks on top of the charts.
**Objective Value Meaning**: The optimal objective value of 27 indicates the maximum total weeks on top of the charts for the selected artists.
**Resource Allocation Summary**: Allocate resources to secure performances from artists 1 and 3, ensuring maximum exposure.
**Implementation Recommendations**: Confirm the availability and contractual agreements with artists 1 and 3, and proceed with marketing and logistical arrangements for their performances.