# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_friendships():
    """Optimize mutual friendships among students using Gurobi"""

    # 1. MODEL & DATA SETUP
    model = gp.Model("maximize_mutual_friendships")
    
    # Assume we have 3 students for simplicity
    n_students = 3
    
    # 2. VARIABLES
    # Binary decision variables for friendships
    x = model.addVars(n_students, n_students, vtype=GRB.BINARY, name="x")
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the number of mutual friendships
    model.setObjective(gp.quicksum(x[i, j] + x[j, i] for i in range(n_students) for j in range(i+1, n_students)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Maximum Friends Constraint: Each student can have at most 5 friends
    model.addConstrs((gp.quicksum(x[i, j] for j in range(n_students)) <= 5 for i in range(n_students)), name="max_friends")
    
    # Mutual Friendship Constraint: Friendships must be mutual
    model.addConstrs((x[i, j] == x[j, i] for i in range(n_students) for j in range(n_students)), name="mutual_friendship")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_students):
            for j in range(n_students):
                if x[i, j].x > 0.5:  # Binary variable, so check if it's 1
                    print(f"Student {i} considers Student {j} a friend")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_friendships()