# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_social_network_influence():
    """Optimize the targeting strategy for maximum influence spread within budget constraints."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("network_2")
    
    # Data: Influence values and costs for each person
    influence_values = [1.5, 0.9, 1.2]
    targeting_costs = [300.0, 150.0, 250.0]
    budget = 500.0
    n_people = len(influence_values)
    
    # CRITICAL: Validate array lengths before loops
    assert len(influence_values) == len(targeting_costs) == n_people, "Array length mismatch"
    
    # 2. VARIABLES
    # Binary decision variables for targeting each person
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{i}") for i in range(n_people)}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total influence
    model.setObjective(gp.quicksum(influence_values[i] * x[i] for i in range(n_people)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Budget constraint
    model.addConstr(gp.quicksum(targeting_costs[i] * x[i] for i in range(n_people)) <= budget, name="budget_constraint")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_people):
            if x[i].x > 1e-6:
                print(f"Person {i+1} is targeted.")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_social_network_influence()