# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_journalist_event_assignment():
    """Optimize the assignment of journalists to events to maximize attendance."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="news_report")
    
    # Data: Event attendance and journalist-event assignments
    event_attendance = [150, 250, 100]  # Attendance for events 1, 2, 3
    journalist_event_pairs = [(1, 1), (1, 2), (2, 2), (2, 3), (3, 1)]
    
    # Validate array lengths to prevent IndexError
    assert len(event_attendance) == 3, "Event attendance array length mismatch"
    
    # Safe range for indexing
    safe_range = range(len(journalist_event_pairs))
    
    # 2. VARIABLES
    # Binary decision variables for journalist-event assignments
    x = {(i, j): mdl.binary_var(name=f"x_{i}_{j}") for i, j in journalist_event_pairs}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total attendance at events covered by journalists
    objective = mdl.sum(event_attendance[j-1] * x[i, j] for i, j in journalist_event_pairs)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    
    # Each journalist can cover up to 2 events
    mdl.add_constraint(x[1, 1] + x[1, 2] <= 2, ctname="journalist_1_limit")
    mdl.add_constraint(x[2, 2] + x[2, 3] <= 2, ctname="journalist_2_limit")
    mdl.add_constraint(x[3, 1] <= 2, ctname="journalist_3_limit")
    
    # Each event can be covered by up to 2 journalists
    mdl.add_constraint(x[1, 1] + x[3, 1] <= 2, ctname="event_1_limit")
    mdl.add_constraint(x[1, 2] + x[2, 2] <= 2, ctname="event_2_limit")
    mdl.add_constraint(x[2, 3] <= 2, ctname="event_3_limit")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i, j in journalist_event_pairs:
            value = solution.get_value(x[i, j])
            if value > 1e-6:
                print(f"x[{i},{j}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
optimize_journalist_event_assignment()