# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_orchestra_performances():
    """Optimize the number of performances for each show to maximize attendance."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("orchestra_optimization")
    
    # Data from the problem statement
    show_ids = [101, 102, 103]
    attendances = {101: 150, 102: 250, 103: 350}
    max_performances = {101: 5, 102: 10, 103: 15}
    capacities = {101: 200, 102: 300, 103: 400}
    total_conductor_availability = 20
    
    # CRITICAL: Validate array lengths before loops
    assert len(show_ids) == len(attendances) == len(max_performances) == len(capacities), "Array length mismatch"
    
    # 2. VARIABLES
    performances = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"P_{i}", lb=0) for i in show_ids}
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(gp.quicksum(attendances[i] * performances[i] for i in show_ids), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total availability of conductors
    model.addConstr(gp.quicksum(performances[i] for i in show_ids) <= total_conductor_availability, name="conductor_availability")
    
    # Orchestra performance limits
    for i in show_ids:
        model.addConstr(performances[i] <= max_performances[i], name=f"max_performances_{i}")
    
    # Venue capacity constraints
    for i in show_ids:
        model.addConstr(attendances[i] * performances[i] <= capacities[i], name=f"venue_capacity_{i}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in show_ids:
            if performances[i].x > 1e-6:
                print(f"Performances[{i}] = {performances[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_orchestra_performances()