# Complete GUROBIPY implementation - Retry Attempt 2

import gurobipy as gp
from gurobipy import GRB

def optimize_party_people():
    # 1. MODEL & DATA SETUP
    model = gp.Model("party_people")
    
    # Sample data based on the provided schema and realistic data
    member_ids = [1, 2, 3]
    event_ids = [101, 102, 103]
    
    # Effectiveness scores
    effectiveness_scores = {
        (1, 101): 0.85,
        (2, 102): 0.95,
        (3, 103): 0.75
    }
    
    # Member availability
    member_availability = {
        (1, 101): True,
        (2, 102): False,
        (3, 103): True
    }
    
    # Event capacities
    max_members_per_event = {
        101: 50,
        102: 100,
        103: 150
    }
    
    # CRITICAL: Validate array lengths before loops
    assert len(effectiveness_scores) == len(member_availability), "Array length mismatch"
    
    # 2. VARIABLES
    # Decision variables: x[i, j] = 1 if member i is assigned to event j, 0 otherwise
    x = model.addVars(member_ids, event_ids, vtype=GRB.BINARY, name="x")
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total effectiveness
    model.setObjective(
        gp.quicksum(effectiveness_scores[i, j] * x[i, j] for i in member_ids for j in event_ids if (i, j) in effectiveness_scores),
        GRB.MAXIMIZE
    )
    
    # 4. CONSTRAINTS
    # Event capacity constraints
    model.addConstrs(
        (gp.quicksum(x[i, j] for i in member_ids) <= max_members_per_event[j] for j in event_ids),
        name="event_capacity"
    )
    
    # Member availability constraints
    model.addConstrs(
        (x[i, j] <= member_availability[i, j] for i in member_ids for j in event_ids if (i, j) in member_availability),
        name="member_availability"
    )
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in member_ids:
            for j in event_ids:
                if x[i, j].x > 0.5:  # Binary variable, so check if it's 1
                    print(f"Member {i} assigned to Event {j}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")

# Run the optimization
optimize_party_people()