# Complete PYOMO implementation - Retry Attempt 2

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def party_people_optimization():
    """Optimize member assignments to maximize event effectiveness."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Example data based on the provided schema and realistic data
    members = [1, 2, 3]
    events = [101, 102, 103]
    effectiveness_scores = {
        (1, 101): 0.85,
        (2, 102): 0.95,
        (3, 103): 0.75
    }
    member_availability = {
        (1, 101): True,
        (2, 102): False,
        (3, 103): True
    }
    max_members_per_event = {
        101: 50,
        102: 100,
        103: 150
    }
    
    # 3. SETS
    model.M = pyo.Set(initialize=members)
    model.E = pyo.Set(initialize=events)
    
    # 4. PARAMETERS
    model.efficiency_score = pyo.Param(model.M, model.E, initialize=effectiveness_scores, default=0)
    model.is_available = pyo.Param(model.M, model.E, initialize=member_availability, within=pyo.Boolean)
    model.max_members_per_event = pyo.Param(model.E, initialize=max_members_per_event)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.M, model.E, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def objective_rule(model):
        return sum(model.efficiency_score[m, e] * model.x[m, e] for m in model.M for e in model.E)
    model.objective = pyo.Objective(rule=objective_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    # Event Capacity Constraints
    def event_capacity_rule(model, e):
        return sum(model.x[m, e] for m in model.M) <= model.max_members_per_event[e]
    model.event_capacity_constraint = pyo.Constraint(model.E, rule=event_capacity_rule)
    
    # Member Availability Constraints
    def member_availability_rule(model, m, e):
        return model.x[m, e] <= model.is_available[m, e]
    model.member_availability_constraint = pyo.Constraint(model.M, model.E, rule=member_availability_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Solve the model
    results = solver.solve(model, tee=True)
    
    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective):.2f}")
        
        # Extract variable values
        print("\nAssignment decisions:")
        for m in model.M:
            for e in model.E:
                if pyo.value(model.x[m, e]) > 0.5:  # Only print assigned members
                    print(f"Member {m} assigned to Event {e}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")

# Execute the optimization
party_people_optimization()