# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_resource_allocation():
    """Optimize resource allocation to minimize casualties within budget constraints."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("resource_allocation")
    
    # Data from the problem statement
    killed = [10, 5, 20]
    injured = [50, 30, 100]
    costs = [2000, 1500, 2500]
    total_budget = 1000000  # Example budget value
    
    # Validate array lengths
    n_locations = len(killed)
    assert len(injured) == len(costs) == n_locations, "Array length mismatch"
    
    # 2. VARIABLES
    # Decision variables for resource allocation
    x = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i}", lb=0) 
         for i in range(n_locations)}
    
    # 3. OBJECTIVE FUNCTION
    # Minimize total casualties
    model.setObjective(gp.quicksum((killed[i] + injured[i]) * x[i] for i in range(n_locations)), GRB.MINIMIZE)
    
    # 4. CONSTRAINTS
    # Budget constraint
    model.addConstr(gp.quicksum(costs[i] * x[i] for i in range(n_locations)) <= total_budget, name="budget_constraint")
    
    # Non-negativity constraints are implicitly handled by the variable lower bounds
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_locations):
            if x[i].x > 1e-6:
                print(f"x[{i}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_resource_allocation()