# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_phone_production(total_ram, total_rom, total_screen):
    """Optimize phone production to maximize profit given resource constraints."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("phone_production")
    
    # Data from the database
    model_ids = [1, 2, 3]
    ram_requirements = [4, 8, 16]
    rom_requirements = [16, 32, 64]
    screen_requirements = [1, 2, 3]
    profit_per_unit = [50.0, 75.0, 100.0]
    
    # CRITICAL: Validate array lengths before loops
    assert len(model_ids) == len(ram_requirements) == len(rom_requirements) == len(screen_requirements) == len(profit_per_unit), "Array length mismatch"
    
    # 2. VARIABLES
    # Variable dictionaries for units produced
    units_produced = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i}", lb=0) for i in model_ids}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total profit
    model.setObjective(gp.quicksum(profit_per_unit[i-1] * units_produced[i] for i in model_ids), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # RAM Constraint
    model.addConstr(gp.quicksum(ram_requirements[i-1] * units_produced[i] for i in model_ids) <= total_ram, name="RAM_constraint")
    
    # ROM Constraint
    model.addConstr(gp.quicksum(rom_requirements[i-1] * units_produced[i] for i in model_ids) <= total_rom, name="ROM_constraint")
    
    # Screen Constraint
    model.addConstr(gp.quicksum(screen_requirements[i-1] * units_produced[i] for i in model_ids) <= total_screen, name="Screen_constraint")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in model_ids:
            if units_produced[i].x > 1e-6:
                print(f"Units produced for model {i}: {units_produced[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Example usage with total available resources
total_available_ram = 10000  # Example value
total_available_rom = 20000  # Example value
total_available_screen = 5000  # Example value

optimize_phone_production(total_available_ram, total_available_rom, total_available_screen)