# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_protein_institute():
    """Optimize resource allocation to maximize sequence identity"""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("protein_institute")
    
    # Data: Sequence identity coefficients and building capacities
    sequence_identity_coefficients = [0.85, 0.9, 0.8]
    building_capacities = [600, 700, 500]
    total_resources_available = 600
    
    # Validate array lengths
    n_institutions = len(sequence_identity_coefficients)
    assert len(building_capacities) == n_institutions, "Array length mismatch"
    
    # 2. VARIABLES
    # Decision variables for resource allocation
    x = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i+1}", lb=0) 
         for i in range(n_institutions)}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total sequence identity
    model.setObjective(gp.quicksum(sequence_identity_coefficients[i] * x[i] 
                                   for i in range(n_institutions)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    # Total resource constraint
    model.addConstr(gp.quicksum(x[i] for i in range(n_institutions)) <= total_resources_available, 
                    name="total_resource_constraint")
    
    # Building capacity constraints
    for i in range(n_institutions):
        model.addConstr(x[i] <= building_capacities[i], name=f"building_capacity_{i+1}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_institutions):
            if x[i].x > 1e-6:
                print(f"x[{i+1}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_protein_institute()