# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_race_track():
    """Optimize the number of races to maximize spectators"""

    # 1. MODEL & DATA SETUP
    model = gp.Model("race_track")
    
    # Data from the problem statement
    seating = [5000, 10000, 15000]
    max_spectators = [4500, 9500, 14500]
    total_races_limit = 10
    n_tracks = len(seating)
    
    # CRITICAL: Validate array lengths before loops
    assert len(seating) == len(max_spectators) == n_tracks, "Array length mismatch"
    
    # 2. VARIABLES
    # Variable dictionaries for number of races at each track
    races = {i: model.addVar(vtype=GRB.INTEGER, name=f"races_{i}", lb=0) 
             for i in range(n_tracks)}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total number of spectators
    model.setObjective(gp.quicksum(seating[i] * races[i] for i in range(n_tracks)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total number of races constraint
    model.addConstr(gp.quicksum(races[i] for i in range(n_tracks)) <= total_races_limit, name="total_races_limit")
    
    # Maximum spectators constraint for each track
    for i in range(n_tracks):
        model.addConstr(seating[i] * races[i] <= max_spectators[i], name=f"max_spectators_{i}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_tracks):
            if races[i].x > 1e-6:
                print(f"races[{i}] = {races[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_race_track()