## 4. Mathematical Optimization Formulation

#### Decision Variables
- Let \( x_{ij} \) be a binary decision variable where \( x_{ij} = 1 \) if driver \( i \) is assigned to school \( j \), and \( x_{ij} = 0 \) otherwise.

#### Objective Function
Minimize the total experience mismatch across all driver-school assignments:

\[
\text{Minimize} \quad \sum_{i,j} \text{YearsMismatch}_{ij} \times x_{ij}
\]

Where:
- \(\text{YearsMismatch}_{ij}\) is the mismatch in years of experience between driver \( i \) and school \( j \), sourced from the `ExperienceMismatch` table.

#### Constraints
1. **Each driver can be assigned to at most one school:**

\[
\sum_{j} x_{ij} \leq 1 \quad \forall i \text{ such that } \text{IsFullTime}_i = \text{True}
\]

Where:
- \(\text{IsFullTime}_i\) is a boolean indicating if driver \( i \) is full-time, sourced from the `Drivers` table.

2. **Each school must receive the exact number of drivers it requires:**

\[
\sum_{i} x_{ij} = \text{RequiredDrivers}_j \quad \forall j
\]

Where:
- \(\text{RequiredDrivers}_j\) is the number of drivers required by school \( j \), sourced from the `SchoolRequirements` table.

#### Data Source Verification:
- \(\text{YearsMismatch}_{ij}\) is sourced from `ExperienceMismatch.YearsMismatch`.
- \(\text{IsFullTime}_i\) is sourced from `Drivers.IsFullTime`.
- \(\text{RequiredDrivers}_j\) is sourced from `SchoolRequirements.RequiredDrivers`.

This formulation ensures that the assignment of drivers to schools minimizes the total experience mismatch while satisfying all operational constraints. The model is linear and can be solved using standard linear programming or mixed-integer programming solvers.

## 5. Gurobipy Implementation

```python
# Complete GUROBIPY implementation - Retry Attempt 4

import gurobipy as gp
from gurobipy import GRB

def school_bus_optimization():
    """Optimize school bus driver assignments to minimize experience mismatch."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("school_bus")
    
    # Data from the database schema
    drivers = [1, 2, 3]
    schools = [101, 102, 103]
    
    # Experience mismatch data
    years_mismatch = {
        (1, 101): 2,
        (2, 102): 1,
        (3, 103): 0
    }
    
    # Full-time status of drivers
    is_full_time = {
        1: True,
        2: True,
        3: True
    }
    
    # School requirements
    required_drivers = {
        101: 3,
        102: 4,
        103: 5
    }
    
    # Validate data lengths
    assert len(drivers) == len(is_full_time), "Driver data length mismatch"
    assert len(schools) == len(required_drivers), "School data length mismatch"
    
    # 2. VARIABLES
    x = model.addVars(drivers, schools, vtype=GRB.BINARY, name="x")
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(
        gp.quicksum(years_mismatch.get((i, j), 0) * x[i, j] for i in drivers for j in schools),
        GRB.MINIMIZE
    )
    
    # 4. CONSTRAINTS
    
    # Each driver can be assigned to at most one school
    for i in drivers:
        if is_full_time[i]:
            model.addConstr(gp.quicksum(x[i, j] for j in schools) <= 1, name=f"driver_{i}_assignment")
    
    # Each school must receive the exact number of drivers it requires
    for j in schools:
        model.addConstr(gp.quicksum(x[i, j] for i in drivers) == required_drivers[j], name=f"school_{j}_requirement")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in drivers:
            for j in schools:
                if x[i, j].x > 0.5:  # Binary variable threshold
                    print(f"Driver {i} assigned to School {j}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    else:
        print("Optimization ended with status", model.status)

# Run the optimization
school_bus_optimization()
```

### Execution Results
**Status**: INFEASIBLE
**Error**: Problem is infeasible
**Analysis**: Gurobipy consistently reports the problem as infeasible, indicating a potential issue with the model constraints or data.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def school_bus_optimization():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="school_bus")
    
    # Data from the problem context
    drivers = [1, 2, 3]
    schools = [101, 102, 103]
    is_full_time = {1: True, 2: True, 3: True}
    required_drivers = {101: 3, 102: 4, 103: 5}
    years_mismatch = {
        (1, 101): 2, (1, 102): 0, (1, 103): 0,
        (2, 101): 0, (2, 102): 1, (2, 103): 0,
        (3, 101): 0, (3, 102): 0, (3, 103): 0
    }
    
    # Validate data lengths
    assert len(drivers) == len(is_full_time)
    assert len(schools) == len(required_drivers)
    
    # 2. VARIABLES
    x = {(i, j): mdl.binary_var(name=f"x_{i}_{j}") for i in drivers for j in schools}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(years_mismatch[i, j] * x[i, j] for i in drivers for j in schools)
    mdl.minimize(objective)
    
    # 4. CONSTRAINTS
    # Each driver can be assigned to at most one school
    for i in drivers:
        if is_full_time[i]:
            mdl.add_constraint(mdl.sum(x[i, j] for j in schools) <= 1, ctname=f"driver_{i}_assignment")
    
    # Each school must receive the exact number of drivers it requires
    for j in schools:
        mdl.add_constraint(mdl.sum(x[i, j] for i in drivers) == required_drivers[j], ctname=f"school_{j}_requirement")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in drivers:
            for j in schools:
                if solution.get_value(x[i, j]) > 0.5:  # Binary variable threshold
                    print(f"Driver {i} assigned to School {j}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

    return mdl

# Run the optimization
school_bus_optimization()
```

### Execution Results
**Status**: INFEASIBLE
**Error**: Problem is infeasible
**Analysis**: DOCplex also reports infeasibility, suggesting that the issue is likely with the model formulation or input data rather than the solver.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation - Retry Attempt 4

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def school_bus_optimization():
    """Optimize school bus driver assignments to minimize experience mismatch."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Example data based on the problem context
    drivers = [1, 2, 3]
    schools = [101, 102, 103]
    is_full_time = {1: True, 2: True, 3: True}
    required_drivers = {101: 3, 102: 4, 103: 5}
    years_mismatch = {
        (1, 101): 2, (1, 102): 3, (1, 103): 4,
        (2, 101): 5, (2, 102): 1, (2, 103): 2,
        (3, 101): 3, (3, 102): 4, (3, 103): 0
    }
    
    # CRITICAL: Validate data lengths
    assert len(drivers) == len(is_full_time), "Driver data length mismatch"
    assert len(schools) == len(required_drivers), "School data length mismatch"
    
    # 3. SETS
    model.D = pyo.Set(initialize=drivers)
    model.S = pyo.Set(initialize=schools)
    
    # 4. PARAMETERS
    model.is_full_time = pyo.Param(model.D, initialize=is_full_time)
    model.required_drivers = pyo.Param(model.S, initialize=required_drivers)
    model.years_mismatch = pyo.Param(model.D * model.S, initialize=years_mismatch)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.D, model.S, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.years_mismatch[i, j] * model.x[i, j] for i in model.D for j in model.S)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.minimize)
    
    # 7. CONSTRAINTS
    
    # Each driver can be assigned to at most one school
    def driver_assignment_rule(model, i):
        if model.is_full_time[i]:
            return sum(model.x[i, j] for j in model.S) <= 1
        else:
            return pyo.Constraint.Skip
    model.driver_assignment_constraint = pyo.Constraint(model.D, rule=driver_assignment_rule)
    
    # Each school must receive the exact number of drivers it requires
    def school_requirement_rule(model, j):
        return sum(model.x[i, j] for i in model.D) == model.required_drivers[j]
    model.school_requirement_constraint = pyo.Constraint(model.S, rule=school_requirement_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)
    
    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nDriver assignments:")
        for i in model.D:
            for j in model.S:
                if pyo.value(model.x[i, j]) > 0.5:  # Binary variable check
                    print(f"Driver {i} assigned to School {j}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Execute the optimization
school_bus_optimization()
```

### Execution Results
**Status**: INFEASIBLE
**Error**: Problem is infeasible
**Analysis**: Pyomo's infeasibility result aligns with the other solvers, reinforcing the likelihood of a model or data issue.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | INFEASIBLE | N/A | 0.55s | N/A | 4 |
| Docplex | INFEASIBLE | N/A | 4.88s | N/A | 4 |
| Pyomo | INFEASIBLE | N/A | 3.62s | N/A | 4 |

### Solver Consistency Analysis
**Result**: All solvers produced consistent results ✓
**Consistent Solvers**: gurobipy, docplex, pyomo
**Solver Retry Summary**: gurobipy: 4 attempts, docplex: 4 attempts, pyomo: 4 attempts

### Final Recommendation
**Confidence Level**: LOW
**Preferred Solver(s)**: multiple
**Reasoning**: All solvers consistently report infeasibility, suggesting the issue is not solver-specific but related to the model or data.

### Business Interpretation
**Overall Strategy**: The current model cannot provide a feasible solution, indicating a need to revisit the data and constraints.
**Objective Value Meaning**: The objective value represents the total experience mismatch, which should be minimized for optimal driver-school assignments.
**Resource Allocation Summary**: Currently, no feasible allocation of drivers to schools meets all constraints.
**Implementation Recommendations**: Review and verify the input data for accuracy, particularly the ExperienceMismatch, Drivers, and SchoolRequirements tables. Ensure that the constraints accurately reflect the business requirements and consider relaxing constraints if necessary.