# Complete Optimization Problem and Solution: singer

## 1. Problem Context and Goals

### Context  
A music production company is focused on enhancing its total sales by strategically promoting a selection of songs. The decision to promote each song is represented by a binary choice: either a song is promoted or it is not. The primary objective is to maximize the overall sales increase, which is directly linked to the potential sales boost each song could achieve if promoted. The company operates under specific constraints, including a fixed budget allocated for promotional activities and a cap on the number of songs that can be promoted. These constraints ensure that the promotion strategy remains financially viable and strategically focused. The budget and the maximum number of songs to promote are critical parameters that guide the decision-making process, ensuring that the company optimizes its promotional efforts within these limits.

### Goals  
The overarching goal of the optimization process is to maximize the total sales of the songs. This is achieved by selecting the optimal set of songs to promote, thereby maximizing the potential sales increase. The success of this strategy is measured by the total sales increase, which is calculated as the sum of the potential sales increases for all promoted songs. The objective is clearly defined in linear terms, focusing on maximizing the sum of potential sales increases, which aligns with the company's strategic aim to enhance revenue through targeted promotions.

## 2. Constraints    

The company faces two primary constraints in its promotional strategy. First, the total cost of promoting the selected songs must not exceed the allocated budget. This ensures that the promotional activities remain within the financial limits set by the company. Second, there is a restriction on the number of songs that can be promoted, which encourages a strategic selection process to maximize the impact of the promotions. These constraints are expressed in linear terms, focusing on the sum of promotion costs and the count of promoted songs, respectively, ensuring that the optimization problem remains linear and manageable.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 1 Database Schema
-- Objective: Schema changes include creating new tables for missing optimization data, modifying existing tables to fill mapping gaps, and updating business configuration logic for scalar parameters and formulas.

CREATE TABLE Song (
  SongID INTEGER,
  PotentialSalesIncrease FLOAT
);

CREATE TABLE PromotionCost (
  SongID INTEGER,
  Cost FLOAT,
  Promote BOOLEAN
);
```

### Data Dictionary  
The data is organized into two main tables, each serving a distinct purpose in the optimization process. The "Song" table contains information about each song, specifically focusing on the potential sales increase that could be achieved if the song is promoted. This data is crucial for determining the objective coefficients in the optimization model. The "PromotionCost" table records the cost associated with promoting each song and indicates whether a song is currently promoted. This information is vital for defining the constraints related to promotion costs and decision variables. Each table and column is designed to support the linear optimization formulation, ensuring that the data aligns with the company's strategic goals.

### Current Stored Values  
```sql
-- Iteration 1 Realistic Data
-- Generated by triple expert (business + data + optimization)
-- Values were determined based on typical music promotion costs and potential sales increases observed in the industry, ensuring that the budget and promotion limits are realistic and allow for meaningful decision-making.

-- Realistic data for Song
INSERT INTO Song (SongID, PotentialSalesIncrease) VALUES (1, 1200.0);
INSERT INTO Song (SongID, PotentialSalesIncrease) VALUES (2, 1800.0);
INSERT INTO Song (SongID, PotentialSalesIncrease) VALUES (3, 2500.0);

-- Realistic data for PromotionCost
INSERT INTO PromotionCost (SongID, Cost, Promote) VALUES (1, 600.0, False);
INSERT INTO PromotionCost (SongID, Cost, Promote) VALUES (2, 900.0, False);
INSERT INTO PromotionCost (SongID, Cost, Promote) VALUES (3, 1200.0, False);
```

## 4. Mathematical Optimization Formulation

#### Decision Variables
- Let \( x_i \) be a binary decision variable for each song \( i \), where \( x_i = 1 \) if song \( i \) is promoted, and \( x_i = 0 \) otherwise.

#### Objective Function
Maximize the total potential sales increase from promoting the songs:
\[ \text{Maximize } Z = 1200x_1 + 1800x_2 + 2500x_3 \]

Data Source Verification:
- Coefficients for the objective function are derived from the `Song.PotentialSalesIncrease` column.

#### Constraints
1. **Budget Constraint**: The total cost of promoting the selected songs must not exceed the allocated budget.
   \[ 600x_1 + 900x_2 + 1200x_3 \leq \text{Budget} \]

2. **Promotion Limit Constraint**: The number of songs promoted must not exceed the maximum allowed.
   \[ x_1 + x_2 + x_3 \leq \text{MaxSongs} \]

Data Source Verification:
- Coefficients for the budget constraint are derived from the `PromotionCost.Cost` column.
- The right-hand side constants for the constraints (Budget and MaxSongs) are parameters that need to be specified based on business configuration or input.

This linear model can be solved using standard linear programming or mixed-integer programming solvers, given specific values for the budget and maximum number of songs to promote.

## 5. Gurobipy Implementation

```python
# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_song_promotion(budget, max_songs):
    """Optimize song promotion strategy to maximize sales increase."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("song_promotion")
    
    # Data from the database
    potential_sales_increase = [1200.0, 1800.0, 2500.0]
    promotion_costs = [600.0, 900.0, 1200.0]
    n_songs = len(potential_sales_increase)
    
    # CRITICAL: Validate array lengths before loops
    assert len(potential_sales_increase) == len(promotion_costs) == n_songs, "Array length mismatch"
    
    # 2. VARIABLES
    # Binary decision variables for each song
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{i}") for i in range(n_songs)}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total potential sales increase
    model.setObjective(gp.quicksum(potential_sales_increase[i] * x[i] for i in range(n_songs)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Budget constraint
    model.addConstr(gp.quicksum(promotion_costs[i] * x[i] for i in range(n_songs)) <= budget, name="budget_constraint")
    
    # Promotion limit constraint
    model.addConstr(gp.quicksum(x[i] for i in range(n_songs)) <= max_songs, name="promotion_limit")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_songs):
            if x[i].x > 1e-6:
                print(f"Song {i+1} is promoted.")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Example usage
budget = 1500.0  # Example budget
max_songs = 2    # Example maximum number of songs to promote
optimize_song_promotion(budget, max_songs)
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 3000.0
**Execution Time**: 0.36 seconds
**Reliability**: high
**Analysis**: Gurobipy found an optimal solution quickly with a consistent optimal value of 3000.0, indicating reliable performance.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_song_promotion(budget, max_songs):
    """Optimize song promotion to maximize sales increase within budget and song limit constraints."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="song_promotion")
    
    # Data from the problem statement
    potential_sales_increase = [1200.0, 1800.0, 2500.0]
    promotion_costs = [600.0, 900.0, 1200.0]
    n_songs = len(potential_sales_increase)
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(potential_sales_increase) == len(promotion_costs) == n_songs, "Array length mismatch"
    safe_range = range(n_songs)  # Safe indexing
    
    # 2. VARIABLES
    # Binary decision variables for each song
    x = {i: mdl.binary_var(name=f"x_{i}") for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total potential sales increase from promoting the songs
    objective = mdl.sum(potential_sales_increase[i] * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Budget constraint
    total_cost = mdl.sum(promotion_costs[i] * x[i] for i in safe_range)
    mdl.add_constraint(total_cost <= budget, ctname="budget_constraint")
    
    # Promotion limit constraint
    total_promoted = mdl.sum(x[i] for i in safe_range)
    mdl.add_constraint(total_promoted <= max_songs, ctname="promotion_limit_constraint")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            if solution.get_value(x[i]) > 0.5:  # Binary variable, check if promoted
                print(f"Song {i+1} is promoted.")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Example usage
optimize_song_promotion(budget=1500, max_songs=2)
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 3000.0
**Execution Time**: 7.28 seconds
**Reliability**: high
**Analysis**: DOCplex also found an optimal solution with the same optimal value as Gurobipy, though it took longer to execute.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def singer_optimization():
    """Optimization model for maximizing song promotion sales increase"""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Data from the problem context
    potential_sales_increase = [1200.0, 1800.0, 2500.0]
    promotion_costs = [600.0, 900.0, 1200.0]
    budget = 2000.0  # Example budget
    max_songs = 2    # Example maximum number of songs to promote
    
    # Validate array lengths
    n_songs = len(potential_sales_increase)
    assert len(promotion_costs) == n_songs, "Array length mismatch"
    
    # 3. SETS
    model.Songs = pyo.RangeSet(1, n_songs)
    
    # 4. PARAMETERS
    model.potential_sales_increase = pyo.Param(model.Songs, initialize={i+1: potential_sales_increase[i] for i in range(n_songs)})
    model.promotion_costs = pyo.Param(model.Songs, initialize={i+1: promotion_costs[i] for i in range(n_songs)})
    
    # 5. VARIABLES
    model.promote = pyo.Var(model.Songs, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def objective_rule(model):
        return sum(model.potential_sales_increase[i] * model.promote[i] for i in model.Songs)
    model.objective = pyo.Objective(rule=objective_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    # Budget constraint
    def budget_constraint_rule(model):
        return sum(model.promotion_costs[i] * model.promote[i] for i in model.Songs) <= budget
    model.budget_constraint = pyo.Constraint(rule=budget_constraint_rule)
    
    # Promotion limit constraint
    def promotion_limit_rule(model):
        return sum(model.promote[i] for i in model.Songs) <= max_songs
    model.promotion_limit = pyo.Constraint(rule=promotion_limit_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)
    
    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nPromotion decisions:")
        for i in model.Songs:
            promote_val = pyo.value(model.promote[i])
            print(f"Song {i}: {'Promote' if promote_val > 0.5 else 'Do not promote'}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Run the optimization
singer_optimization()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 3700.0
**Execution Time**: 2.88 seconds
**Reliability**: medium
**Analysis**: Pyomo reported a different optimal value of 3700.0, which suggests a potential issue with model formulation or solver configuration.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | OPTIMAL | 3000.00 | 0.36s | N/A | N/A |
| Docplex | OPTIMAL | 3000.00 | 7.28s | N/A | N/A |
| Pyomo | OPTIMAL | 3700.00 | 2.88s | N/A | N/A |

### Solver Consistency Analysis
**Result**: Solvers produced inconsistent results
**Consistent Solvers**: gurobipy, docplex
**Inconsistent Solvers**: pyomo
**Potential Issues**:
- Pyomo might have a different interpretation of constraints or objective function.
- Possible data input errors or solver configuration issues in Pyomo.
**Majority Vote Optimal Value**: 3000.0

### Final Recommendation
**Recommended Optimal Value**: 3000.0
**Confidence Level**: HIGH
**Preferred Solver(s)**: gurobipy
**Reasoning**: Gurobipy provided a consistent and quick solution, aligning with DOCplex results, indicating reliability.

### Business Interpretation
**Overall Strategy**: Promoting the selected songs will result in a potential sales increase of 3000 units.
**Objective Value Meaning**: The optimal objective value represents the maximum potential sales increase achievable under the given constraints.
**Resource Allocation Summary**: Allocate resources to promote the songs that maximize the sales increase within the budget and promotion limits.
**Implementation Recommendations**: Verify the input data and constraints, ensure solver configurations are consistent, and proceed with the promotion strategy as per the recommended decision variables.