# Complete Optimization Problem and Solution: soccer_2

## 1. Problem Context and Goals

### Context  
The soccer league is organizing tryouts to select players from various colleges for different positions on a team. The primary decision involves determining whether each player should be selected, represented by a binary decision variable for each player. The objective is to minimize the total number of yellow cards among the selected players. This selection process must ensure that each position on the team is filled with the required number of players. Additionally, to promote diversity, there is a limit on the number of players that can be selected from any single college. The maximum number of players allowed from a single college is a critical parameter in this process. The number of players required for each position is specified to ensure that all positions are adequately filled. These requirements are expressed in a way that naturally aligns with linear optimization, focusing on precise decision-making without involving complex mathematical operations like products or divisions.

### Goals  
The goal of this optimization problem is to minimize the total number of yellow cards among the selected players. The metric used to achieve this goal is the sum of yellow cards for all selected players. Success in this context is measured by achieving the lowest possible total of yellow cards while adhering to the constraints of filling each position and maintaining college diversity. The optimization goal is clearly defined in linear terms, focusing on minimizing the sum of yellow cards associated with the selected players.

## 2. Constraints    

The constraints for this optimization problem are designed to ensure that the team composition meets specific requirements:

- For each position on the team, the number of selected players must match the required number of players for that position. This ensures that all positions are adequately filled.
- To maintain diversity, the number of players selected from any single college must not exceed the maximum allowed. This constraint ensures that no single college is over-represented in the team selection.

These constraints are expressed in business terms that naturally lead to linear mathematical forms, avoiding complex operations like variable products or divisions.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 2 Database Schema
-- Objective: Schema adjustments were made to incorporate missing yellow card data and improve optimization mapping. Configuration logic was updated to include scalar parameters and formulas.

CREATE TABLE Tryout (
  pID INTEGER,
  pPos STRING,
  cName STRING,
  yCard INTEGER
);

CREATE TABLE PositionRequirements (
  position STRING,
  required_players INTEGER
);

CREATE TABLE CollegeLimits (
  college STRING,
  max_players INTEGER
);

CREATE TABLE PlayerYellowCards (
  pID INTEGER,
  yCard INTEGER
);
```

### Data Dictionary  
The data dictionary provides a comprehensive mapping of tables and columns to their business purposes and roles in the optimization process:

- **Tryout Table**: This table stores data about players participating in the tryouts. Each player is identified by a unique ID, and the table includes information about the position they are trying out for, the college they represent, and the number of yellow cards they have received. This table plays a crucial role in defining the decision variables for the optimization problem.

- **PositionRequirements Table**: This table defines the number of players required for each position on the team. It ensures that the constraints related to filling each position are met.

- **CollegeLimits Table**: This table specifies the maximum number of players that can be selected from each college. It is used to enforce diversity in player selection by limiting the representation from any single college.

- **PlayerYellowCards Table**: This table records the number of yellow cards for each player, linking this data to the players' unique IDs. It is used in the objective function to minimize the total number of yellow cards among selected players.

### Current Stored Values  
```sql
-- Iteration 2 Realistic Data
-- Generated by triple expert (business + data + optimization)
-- Values were determined based on typical soccer team compositions and college diversity requirements, ensuring a balance between player skill (yellow cards) and team needs (positions and college representation).

-- Realistic data for Tryout
INSERT INTO Tryout (pID, pPos, cName, yCard) VALUES (1, 'Goalkeeper', 'College A', 0);
INSERT INTO Tryout (pID, pPos, cName, yCard) VALUES (2, 'Defender', 'College B', 1);
INSERT INTO Tryout (pID, pPos, cName, yCard) VALUES (3, 'Midfielder', 'College C', 2);
INSERT INTO Tryout (pID, pPos, cName, yCard) VALUES (4, 'Forward', 'College A', 1);
INSERT INTO Tryout (pID, pPos, cName, yCard) VALUES (5, 'Defender', 'College B', 0);

-- Realistic data for PositionRequirements
INSERT INTO PositionRequirements (position, required_players) VALUES ('Goalkeeper', 1);
INSERT INTO PositionRequirements (position, required_players) VALUES ('Defender', 4);
INSERT INTO PositionRequirements (position, required_players) VALUES ('Midfielder', 4);
INSERT INTO PositionRequirements (position, required_players) VALUES ('Forward', 2);

-- Realistic data for CollegeLimits
INSERT INTO CollegeLimits (college, max_players) VALUES ('College A', 3);
INSERT INTO CollegeLimits (college, max_players) VALUES ('College B', 3);
INSERT INTO CollegeLimits (college, max_players) VALUES ('College C', 3);

-- Realistic data for PlayerYellowCards
INSERT INTO PlayerYellowCards (pID, yCard) VALUES (1, 0);
INSERT INTO PlayerYellowCards (pID, yCard) VALUES (2, 1);
INSERT INTO PlayerYellowCards (pID, yCard) VALUES (3, 2);
```

## 4. Mathematical Optimization Formulation

#### Decision Variables
- Let \( x_i \) be a binary decision variable for each player \( i \), where \( x_i = 1 \) if player \( i \) is selected, and \( x_i = 0 \) otherwise.

#### Objective Function
Minimize the total number of yellow cards among the selected players:
\[ \text{Minimize} \quad \sum_{i} yCard_i \times x_i \]

Data Source Verification:
- \( yCard_i \) is the number of yellow cards for player \( i \), sourced from `Tryout.yCard`.

#### Constraints

1. **Position Requirements:**
   For each position \( p \), the number of selected players must match the required number of players for that position.
   \[
   \sum_{i \in P_p} x_i = \text{required\_players}_p \quad \forall p
   \]
   where \( P_p \) is the set of players trying out for position \( p \).

   Data Source Verification:
   - \( \text{required\_players}_p \) is sourced from `PositionRequirements.required_players`.

2. **College Diversity:**
   The number of players selected from any single college must not exceed the maximum allowed.
   \[
   \sum_{i \in C_c} x_i \leq \text{max\_players}_c \quad \forall c
   \]
   where \( C_c \) is the set of players from college \( c \).

   Data Source Verification:
   - \( \text{max\_players}_c \) is sourced from `CollegeLimits.max_players`.

3. **Binary Decision Variables:**
   \[
   x_i \in \{0, 1\} \quad \forall i
   \]

Data Source Verification:
- Player data, including position and college, is sourced from `Tryout.pPos` and `Tryout.cName`.

This formulation provides a complete and immediately solvable linear mathematical model, ensuring that all positions are filled, college diversity is maintained, and the total number of yellow cards is minimized.

## 5. Gurobipy Implementation

```python
# Complete GUROBIPY implementation - Retry Attempt 4

import gurobipy as gp
from gurobipy import GRB

def soccer_team_selection():
    # 1. MODEL & DATA SETUP
    model = gp.Model("soccer_2")

    # Data from the problem statement
    players = [
        {"pID": 1, "pPos": "Goalkeeper", "cName": "College A", "yCard": 0},
        {"pID": 2, "pPos": "Defender", "cName": "College B", "yCard": 1},
        {"pID": 3, "pPos": "Midfielder", "cName": "College C", "yCard": 2},
        {"pID": 4, "pPos": "Forward", "cName": "College A", "yCard": 1},
        {"pID": 5, "pPos": "Defender", "cName": "College B", "yCard": 0}
    ]

    position_requirements = {
        "Goalkeeper": 1,
        "Defender": 4,
        "Midfielder": 4,
        "Forward": 2
    }

    college_limits = {
        "College A": 3,
        "College B": 3,
        "College C": 3
    }

    # 2. VARIABLES
    x = {player["pID"]: model.addVar(vtype=GRB.BINARY, name=f"x_{player['pID']}")
         for player in players}

    # 3. OBJECTIVE FUNCTION
    model.setObjective(gp.quicksum(player["yCard"] * x[player["pID"]] for player in players), GRB.MINIMIZE)

    # 4. CONSTRAINTS

    # Position requirements
    for position, required_players in position_requirements.items():
        model.addConstr(
            gp.quicksum(x[player["pID"]] for player in players if player["pPos"] == position) == required_players,
            name=f"position_{position}"
        )

    # College diversity
    for college, max_players in college_limits.items():
        model.addConstr(
            gp.quicksum(x[player["pID"]] for player in players if player["cName"] == college) <= max_players,
            name=f"college_{college}"
        )

    # 5. SOLVING & RESULTS
    model.optimize()

    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for player in players:
            if x[player["pID"]].x > 1e-6:
                print(f"Player {player['pID']} selected")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")

    return model

# Run the optimization
soccer_team_selection()
```

### Execution Results
**Status**: INFEASIBLE
**Error**: Problem is infeasible
**Analysis**: Gurobipy consistently reports infeasibility, indicating a potential issue with the constraints or data.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def soccer_team_selection():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="soccer_team_selection")

    # Data from the problem statement
    players = [
        {"pID": 1, "pPos": "Goalkeeper", "cName": "College A", "yCard": 0},
        {"pID": 2, "pPos": "Defender", "cName": "College B", "yCard": 1},
        {"pID": 3, "pPos": "Midfielder", "cName": "College C", "yCard": 2},
        {"pID": 4, "pPos": "Forward", "cName": "College A", "yCard": 1},
        {"pID": 5, "pPos": "Defender", "cName": "College B", "yCard": 0}
    ]

    position_requirements = {
        "Goalkeeper": 1,
        "Defender": 4,
        "Midfielder": 4,
        "Forward": 2
    }

    college_limits = {
        "College A": 3,
        "College B": 3,
        "College C": 3
    }

    # 2. VARIABLES
    x = {player["pID"]: mdl.binary_var(name=f"x_{player['pID']}") for player in players}

    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(player["yCard"] * x[player["pID"]] for player in players)
    mdl.minimize(objective)

    # 4. CONSTRAINTS

    # Position requirements
    for position, required_players in position_requirements.items():
        mdl.add_constraint(
            mdl.sum(x[player["pID"]] for player in players if player["pPos"] == position) == required_players,
            ctname=f"position_{position}"
        )

    # College diversity
    for college, max_players in college_limits.items():
        mdl.add_constraint(
            mdl.sum(x[player["pID"]] for player in players if player["cName"] == college) <= max_players,
            ctname=f"college_{college}"
        )

    # 5. SOLVING & RESULTS
    solution = mdl.solve()

    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for player in players:
            if solution.get_value(x[player["pID"]]) > 0.5:
                print(f"Player {player['pID']} selected")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

    return mdl

# Execute the function
soccer_team_selection()
```

### Execution Results
**Status**: INFEASIBLE
**Error**: Problem is infeasible
**Analysis**: DOCplex also reports infeasibility, suggesting that the problem constraints cannot be satisfied with the given data.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation - Retry Attempt 4

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def soccer_optimization():
    """Pyomo model for minimizing yellow cards in soccer team selection"""

    # 1. MODEL CREATION
    model = pyo.ConcreteModel()

    # 2. DATA SETUP
    # Player data
    players = [1, 2, 3, 4, 5]
    positions = ['Goalkeeper', 'Defender', 'Midfielder', 'Forward']
    colleges = ['College A', 'College B', 'College C']
    
    # Player attributes
    player_positions = {1: 'Goalkeeper', 2: 'Defender', 3: 'Midfielder', 4: 'Forward', 5: 'Defender'}
    player_colleges = {1: 'College A', 2: 'College B', 3: 'College C', 4: 'College A', 5: 'College B'}
    yellow_cards = {1: 0, 2: 1, 3: 2, 4: 1, 5: 0}

    # Position requirements
    position_requirements = {'Goalkeeper': 1, 'Defender': 4, 'Midfielder': 4, 'Forward': 2}

    # College limits
    college_limits = {'College A': 3, 'College B': 3, 'College C': 3}

    # 3. SETS
    model.Players = pyo.Set(initialize=players)
    model.Positions = pyo.Set(initialize=positions)
    model.Colleges = pyo.Set(initialize=colleges)

    # 4. PARAMETERS
    model.player_positions = pyo.Param(model.Players, initialize=player_positions)
    model.player_colleges = pyo.Param(model.Players, initialize=player_colleges)
    model.yellow_cards = pyo.Param(model.Players, initialize=yellow_cards)
    model.position_requirements = pyo.Param(model.Positions, initialize=position_requirements)
    model.college_limits = pyo.Param(model.Colleges, initialize=college_limits)

    # 5. VARIABLES
    model.x = pyo.Var(model.Players, within=pyo.Binary)

    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.yellow_cards[i] * model.x[i] for i in model.Players)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.minimize)

    # 7. CONSTRAINTS

    # Position requirements
    def position_constraint_rule(model, p):
        return sum(model.x[i] for i in model.Players if model.player_positions[i] == p) == model.position_requirements[p]
    model.position_constraints = pyo.Constraint(model.Positions, rule=position_constraint_rule)

    # College diversity
    def college_constraint_rule(model, c):
        return sum(model.x[i] for i in model.Players if model.player_colleges[i] == c) <= model.college_limits[c]
    model.college_constraints = pyo.Constraint(model.Colleges, rule=college_constraint_rule)

    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')

    # Solve the model
    results = solver.solve(model, tee=True)

    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")

        # Extract variable values
        print("\nSelected players:")
        for i in model.Players:
            if pyo.value(model.x[i]) > 0.5:  # Binary variable, check if selected
                print(f"Player {i} selected")
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")

    return model

# Run the optimization
soccer_optimization()
```

### Execution Results
**Status**: INFEASIBLE
**Error**: Problem is infeasible
**Analysis**: Pyomo confirms infeasibility, aligning with the results from the other solvers.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | INFEASIBLE | N/A | 0.27s | N/A | 4 |
| Docplex | INFEASIBLE | N/A | 2.37s | N/A | 4 |
| Pyomo | INFEASIBLE | N/A | 2.91s | N/A | 4 |

### Solver Consistency Analysis
**Result**: All solvers produced consistent results ✓
**Consistent Solvers**: gurobipy, docplex, pyomo
**Solver Retry Summary**: gurobipy: 4 attempts, docplex: 4 attempts, pyomo: 4 attempts

### Final Recommendation
**Confidence Level**: HIGH
**Preferred Solver(s)**: multiple
**Reasoning**: All solvers consistently report infeasibility, indicating a high confidence in the result. The issue likely lies in the problem formulation or data.

### Business Interpretation
**Overall Strategy**: The current constraints and data do not allow for a feasible selection of players that meets all requirements.
**Objective Value Meaning**: The objective value represents the total number of yellow cards among selected players, which cannot be minimized due to infeasibility.
**Resource Allocation Summary**: Re-evaluate the constraints and data to ensure they accurately reflect the team's needs and available resources.
**Implementation Recommendations**: Review and adjust the position requirements and college diversity constraints. Verify the accuracy of the input data.