# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def storm_record_optimization():
    """Optimize resource allocation to minimize storm damage costs."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("storm_record")
    
    # Data from the problem context
    damage_costs = [12.0, 25.0, 18.0]
    max_speeds = [110.0, 140.0, 130.0]
    num_cities_affected = [4, 6, 5]
    total_resources_available = 600.0
    
    # CRITICAL: Validate array lengths before loops
    n_regions = len(damage_costs)
    assert len(max_speeds) == len(num_cities_affected) == n_regions, "Array length mismatch"
    
    # 2. VARIABLES
    # Variable dictionaries for resource allocation
    x = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i}", lb=0) 
         for i in range(n_regions)}
    
    # 3. OBJECTIVE FUNCTION
    # Minimize the total damage cost
    model.setObjective(gp.quicksum(damage_costs[i] * x[i] for i in range(n_regions)), GRB.MINIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total resource availability constraint
    model.addConstr(gp.quicksum(x[i] for i in range(n_regions)) <= total_resources_available, name="total_resource_limit")
    
    # Resource allocation constraints based on storm severity
    for i in range(n_regions):
        model.addConstr(x[i] <= num_cities_affected[i] * max_speeds[i], name=f"severity_limit_{i}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_regions):
            if x[i].x > 1e-6:
                print(f"x[{i}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
storm_record_optimization()