# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_promotional_tweets():
    """Optimize the allocation of promotional tweets to maximize engagement."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("twitter_1")
    
    # Example data setup
    user_ids = [1, 2, 3]
    engagement_coefficients = [0.6, 0.8, 0.4]
    follower_limit = 2
    min_engagement_threshold = 0.5
    
    # CRITICAL: Validate array lengths before loops
    assert len(user_ids) == len(engagement_coefficients), "Array length mismatch"
    
    # 2. VARIABLES
    # Binary decision variables for each user
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{i}") for i in user_ids}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total engagement score
    model.setObjective(gp.quicksum(engagement_coefficients[i-1] * x[i] for i in user_ids), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total number of users constraint
    model.addConstr(gp.quicksum(x[i] for i in user_ids) <= follower_limit, name="follower_limit")
    
    # Minimum engagement score constraint
    for i in user_ids:
        model.addConstr(x[i] * engagement_coefficients[i-1] >= min_engagement_threshold * x[i], name=f"min_engagement_{i}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in user_ids:
            if x[i].x > 1e-6:
                print(f"x[{i}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_promotional_tweets()