## 4. Mathematical Optimization Formulation

#### Decision Variables
- Let \( x_i \) be a binary decision variable for each user \( i \), where \( x_i = 1 \) if user \( i \) receives the promotional tweet, and \( x_i = 0 \) otherwise.

#### Objective Function
- Maximize the total engagement score across all users:
  \[
  \text{Maximize } \sum_{i} \text{engagement\_coefficient}_i \times x_i
  \]
  where \(\text{engagement\_coefficient}_i\) is the engagement coefficient for user \( i \).

#### Constraints
1. **Total Number of Users Constraint**: The total number of users receiving promotional tweets must not exceed the platform's follower limit.
   \[
   \sum_{i} x_i \leq \text{follower\_limit}
   \]
   - Data Source Verification: The follower limit is a business configuration parameter.

2. **Minimum Engagement Score Constraint**: Each user selected must have an engagement score that meets or exceeds the minimum engagement threshold.
   \[
   x_i \times \text{engagement\_coefficient}_i \geq \text{min\_engagement\_threshold} \times x_i \quad \forall i
   \]
   - Data Source Verification: \(\text{engagement\_coefficient}_i\) from `user_profiles.engagement_coefficient` or `engagement_coefficients.coefficient`.
   - The minimum engagement threshold is a business configuration parameter.

3. **Binary Decision Variable Constraint**: Each decision variable is binary.
   \[
   x_i \in \{0, 1\} \quad \forall i
   \]

Data Source Verification:
- \(\text{engagement\_coefficient}_i\) is sourced from `user_profiles.engagement_coefficient` or `engagement_coefficients.coefficient`.
- The follower limit and minimum engagement threshold are derived from business configuration parameters.

This formulation provides a complete and immediately solvable linear mathematical model using the provided data and constraints.

## 5. Gurobipy Implementation

```python
# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_promotional_tweets():
    """Optimize the allocation of promotional tweets to maximize engagement."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("twitter_1")
    
    # Example data setup
    user_ids = [1, 2, 3]
    engagement_coefficients = [0.6, 0.8, 0.4]
    follower_limit = 2
    min_engagement_threshold = 0.5
    
    # CRITICAL: Validate array lengths before loops
    assert len(user_ids) == len(engagement_coefficients), "Array length mismatch"
    
    # 2. VARIABLES
    # Binary decision variables for each user
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{i}") for i in user_ids}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total engagement score
    model.setObjective(gp.quicksum(engagement_coefficients[i-1] * x[i] for i in user_ids), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total number of users constraint
    model.addConstr(gp.quicksum(x[i] for i in user_ids) <= follower_limit, name="follower_limit")
    
    # Minimum engagement score constraint
    for i in user_ids:
        model.addConstr(x[i] * engagement_coefficients[i-1] >= min_engagement_threshold * x[i], name=f"min_engagement_{i}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in user_ids:
            if x[i].x > 1e-6:
                print(f"x[{i}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_promotional_tweets()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 1.4
**Execution Time**: 0.20 seconds
**Reliability**: high
**Analysis**: Gurobipy found an optimal solution quickly with the shortest execution time, indicating efficient performance and reliability.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_promotional_tweets():
    """Optimize the allocation of promotional tweets to maximize engagement."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="twitter_1")
    
    # Data: User engagement coefficients
    user_ids = [1, 2, 3]
    engagement_coefficients = [0.6, 0.8, 0.4]
    
    # Business configuration parameters
    follower_limit = 2
    min_engagement_threshold = 0.5
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(user_ids) == len(engagement_coefficients), "Array length mismatch"
    safe_range = range(len(user_ids))  # Safe indexing
    
    # 2. VARIABLES
    # Binary decision variables for each user
    x = {i: mdl.binary_var(name=f"x_{user_ids[i]}") for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total engagement score
    objective = mdl.sum(engagement_coefficients[i] * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total number of users constraint
    total_users = mdl.sum(x[i] for i in safe_range)
    mdl.add_constraint(total_users <= follower_limit, ctname="follower_limit")
    
    # Minimum engagement score constraint
    for i in safe_range:
        mdl.add_constraint(x[i] * engagement_coefficients[i] >= min_engagement_threshold * x[i], 
                           ctname=f"min_engagement_{user_ids[i]}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            if value > 1e-6:
                print(f"x[{user_ids[i]}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
optimize_promotional_tweets()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 1.4
**Execution Time**: 2.99 seconds
**Reliability**: high
**Analysis**: DOCplex also found an optimal solution, but with a longer execution time compared to the other solvers. Despite this, the result is consistent and reliable.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def optimize_promotional_tweets():
    """Optimize the allocation of promotional tweets to maximize engagement."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Example data based on the problem context
    user_ids = [1, 2, 3]
    engagement_coefficients = [0.6, 0.8, 0.4]
    follower_limit = 2
    min_engagement_threshold = 0.5
    
    # CRITICAL: Validate array lengths before indexing
    assert len(user_ids) == len(engagement_coefficients), "Array length mismatch"
    
    # 3. SETS
    model.I = pyo.Set(initialize=user_ids)
    
    # 4. PARAMETERS
    model.engagement_coefficient = pyo.Param(model.I, initialize={user_ids[i]: engagement_coefficients[i] for i in range(len(user_ids))})
    
    # 5. VARIABLES
    model.x = pyo.Var(model.I, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.engagement_coefficient[i] * model.x[i] for i in model.I)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    
    # Total number of users constraint
    def total_users_rule(model):
        return sum(model.x[i] for i in model.I) <= follower_limit
    model.total_users_constraint = pyo.Constraint(rule=total_users_rule)
    
    # Minimum engagement score constraint
    def min_engagement_rule(model, i):
        return model.x[i] * model.engagement_coefficient[i] >= min_engagement_threshold * model.x[i]
    model.min_engagement_constraint = pyo.Constraint(model.I, rule=min_engagement_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nVariable values:")
        for i in model.I:
            x_val = pyo.value(model.x[i])
            if x_val > 1e-6:  # Only print non-zero values
                print(f"x[{i}] = {int(x_val)}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Run the optimization
optimize_promotional_tweets()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 1.4
**Execution Time**: 0.92 seconds
**Reliability**: high
**Analysis**: Pyomo achieved an optimal solution with a moderate execution time, demonstrating good reliability and consistency with other solvers.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | OPTIMAL | 1.40 | 0.20s | N/A | N/A |
| Docplex | OPTIMAL | 1.40 | 2.99s | N/A | N/A |
| Pyomo | OPTIMAL | 1.40 | 0.92s | N/A | N/A |

### Solver Consistency Analysis
**Result**: All solvers produced consistent results ✓
**Consistent Solvers**: gurobipy, docplex, pyomo
**Majority Vote Optimal Value**: 1.4

### Final Recommendation
**Recommended Optimal Value**: 1.4
**Confidence Level**: HIGH
**Preferred Solver(s)**: gurobipy
**Reasoning**: Gurobipy is preferred due to its efficient execution time and high reliability, making it suitable for scenarios requiring quick decision-making.

### Business Interpretation
**Overall Strategy**: The optimal engagement score of 1.4 indicates the maximum potential engagement achievable under the given constraints.
**Objective Value Meaning**: The optimal objective value of 1.4 reflects the total engagement score achievable by targeting the selected users.
**Resource Allocation Summary**: Resources should be allocated to users with the highest engagement coefficients within the follower limit to maximize engagement.
**Implementation Recommendations**: Implement the solution by targeting users identified by the decision variables, ensuring compliance with the follower limit and engagement thresholds.