# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_basketball_resources():
    """Optimize resource allocation for university basketball teams"""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("university_basketball")
    
    # Data from the problem statement
    team_ids = [1, 2, 3]
    win_percentages = [0.75, 0.6, 0.85]
    min_allocations = [50, 60, 70]
    max_allocations = [200, 180, 190]
    total_resources_available = 400  # Example total resources available
    
    # CRITICAL: Validate array lengths before loops
    assert len(team_ids) == len(win_percentages) == len(min_allocations) == len(max_allocations), "Array length mismatch"
    
    # 2. VARIABLES
    # Variable dictionaries for resource allocation
    resource_allocation = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"resource_allocation_{i}", lb=0) 
                           for i in team_ids}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the sum of the product of each team's win percentage and the resources allocated to them
    model.setObjective(gp.quicksum(win_percentages[i-1] * resource_allocation[i] for i in team_ids), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total resources constraint
    model.addConstr(gp.quicksum(resource_allocation[i] for i in team_ids) <= total_resources_available, name="total_resources")
    
    # Minimum and maximum allocation constraints for each team
    for i in team_ids:
        model.addConstr(resource_allocation[i] >= min_allocations[i-1], name=f"min_allocation_{i}")
        model.addConstr(resource_allocation[i] <= max_allocations[i-1], name=f"max_allocation_{i}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in team_ids:
            if resource_allocation[i].x > 1e-6:
                print(f"Resource allocation for team {i}: {resource_allocation[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_basketball_resources()