# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def optimize_workshop_submissions():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="workshop_paper")
    
    # Data from the database
    scores = [8.5, 6.0, 9.2, 7.3, 5.5]  # Submission scores
    capacities = [50, 100, 150]  # Venue capacities for each workshop
    min_submissions = [3, 5, 7]  # Minimum submissions required for each workshop
    
    # Validate array lengths to prevent IndexError
    n_submissions = len(scores)
    n_workshops = len(capacities)
    assert len(min_submissions) == n_workshops, "Array length mismatch"
    
    # Safe indexing
    safe_range_submissions = range(n_submissions)
    safe_range_workshops = range(n_workshops)
    
    # 2. VARIABLES
    # Binary decision variables for acceptance of submissions
    x = {(i, j): mdl.binary_var(name=f"x_{i}_{j}") for i in safe_range_workshops for j in safe_range_submissions}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total quality score of accepted submissions
    objective = mdl.sum(scores[j] * x[i, j] for i in safe_range_workshops for j in safe_range_submissions)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    # Workshop Capacity Constraints
    for i in safe_range_workshops:
        mdl.add_constraint(mdl.sum(x[i, j] for j in safe_range_submissions) <= capacities[i], ctname=f"capacity_{i}")
    
    # Minimum Submissions Constraints
    for i in safe_range_workshops:
        mdl.add_constraint(mdl.sum(x[i, j] for j in safe_range_submissions) >= min_submissions[i], ctname=f"min_submissions_{i}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range_workshops:
            for j in safe_range_submissions:
                if solution.get_value(x[i, j]) > 0.5:  # Check if submission is accepted
                    print(f"Submission {j} accepted for Workshop {i}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

    return mdl

# Execute the optimization
optimize_workshop_submissions()