# Complete GUROBIPY implementation - Retry Attempt 4

import gurobipy as gp
from gurobipy import GRB

def optimize_workshop_paper():
    # 1. MODEL & DATA SETUP
    model = gp.Model("workshop_paper")
    
    # Data from the database
    scores = [8.5, 6.0, 9.2, 7.3, 5.5]
    capacities = {1: 50, 2: 100, 3: 150}
    min_submissions = {1: 3, 2: 5, 3: 7}
    
    # Number of submissions
    n_submissions = len(scores)
    n_workshops = len(capacities)
    
    # CRITICAL: Validate array lengths before loops
    assert len(scores) == n_submissions, "Array length mismatch for scores"
    
    # 2. VARIABLES
    # Binary decision variables for acceptance of submissions
    x = model.addVars(n_workshops, n_submissions, vtype=GRB.BINARY, name="x")
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total quality score of accepted submissions
    model.setObjective(gp.quicksum(scores[j] * x[i, j] for i in range(n_workshops) for j in range(n_submissions)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Workshop Capacity Constraints
    for i in range(n_workshops):
        model.addConstr(gp.quicksum(x[i, j] for j in range(n_submissions)) <= capacities[i+1], name=f"capacity_{i+1}")
    
    # Minimum Submissions Constraints
    for i in range(n_workshops):
        model.addConstr(gp.quicksum(x[i, j] for j in range(n_submissions)) >= min_submissions[i+1], name=f"min_submissions_{i+1}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_workshops):
            for j in range(n_submissions):
                if x[i, j].x > 1e-6:
                    print(f"x[{i},{j}] = {x[i, j].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_workshop_paper()