# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def wrestler_optimization():
    """Optimize wrestler assignments to maximize days titles are held."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Wrestler data
    wrestler_ids = [1, 2, 3]
    days_held = {1: 30, 2: 45, 3: 60}
    max_events = {1: 5, 2: 6, 3: 7}
    
    # Event data
    event_ids = [101, 102, 103]
    max_participants = {101: 10, 102: 12, 103: 15}
    
    # CRITICAL: Validate data lengths
    assert len(wrestler_ids) == len(days_held) == len(max_events), "Wrestler data length mismatch"
    assert len(event_ids) == len(max_participants), "Event data length mismatch"
    
    # 3. SETS
    model.Wrestlers = pyo.Set(initialize=wrestler_ids)
    model.Events = pyo.Set(initialize=event_ids)
    
    # 4. PARAMETERS
    model.DaysHeld = pyo.Param(model.Wrestlers, initialize=days_held)
    model.MaxEvents = pyo.Param(model.Wrestlers, initialize=max_events)
    model.MaxParticipants = pyo.Param(model.Events, initialize=max_participants)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.Wrestlers, model.Events, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.DaysHeld[i] * model.x[i, j] for i in model.Wrestlers for j in model.Events)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    
    # Wrestler Participation Constraints
    def wrestler_participation_rule(model, i):
        return sum(model.x[i, j] for j in model.Events) <= model.MaxEvents[i]
    model.wrestler_participation_constraint = pyo.Constraint(model.Wrestlers, rule=wrestler_participation_rule)
    
    # Event Capacity Constraints
    def event_capacity_rule(model, j):
        return sum(model.x[i, j] for i in model.Wrestlers) <= model.MaxParticipants[j]
    model.event_capacity_constraint = pyo.Constraint(model.Events, rule=event_capacity_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)
    
    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nAssignment of wrestlers to events:")
        for i in model.Wrestlers:
            for j in model.Events:
                if pyo.value(model.x[i, j]) > 0.5:  # Only print assignments
                    print(f"Wrestler {i} assigned to Event {j}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Run the optimization
wrestler_optimization()