# Complete Optimization Problem and Solution: aircraft

## 1. Problem Context and Goals

### Context  
The business problem involves optimizing the assignment of aircraft to airports to minimize total operational costs while ensuring that each airport's passenger demand is met and aircraft capacity constraints are respected. The decision to be made is the number of each type of aircraft assigned to each airport, which directly impacts operational costs. Each aircraft type has a specific passenger capacity, and each airport has a defined passenger demand and a maximum number of aircraft movements it can handle. The operational cost of assigning a specific aircraft type to a particular airport is known and varies based on the aircraft and airport combination. The business configuration includes a maximum number of aircraft movements allowed at each airport, which ensures that the airport's movement capacity is not exceeded. The total operational cost is calculated as the sum of the costs of all aircraft assignments, where each cost is the product of the operational cost for a specific aircraft-airport pair and the number of aircraft assigned to that pair.

### Goals  
The primary goal of this optimization is to minimize the total operational costs associated with assigning aircraft to airports. Success is measured by achieving the lowest possible sum of operational costs across all aircraft assignments while ensuring that all airports meet their passenger demand and do not exceed their aircraft movement capacity. The optimization focuses on making precise operational decisions that lead to a linear formulation, avoiding any scenarios that would require nonlinear relationships such as variable products or divisions.

## 2. Constraints    

The optimization must adhere to the following constraints:  
1. **Passenger Demand Constraint**: For each airport, the total passenger capacity of all assigned aircraft must meet or exceed the airport's passenger demand. This ensures that the airport can accommodate all passengers.  
2. **Aircraft Movement Constraint**: For each airport, the total number of assigned aircraft must not exceed the airport's maximum number of aircraft movements. This ensures that the airport's movement capacity is not exceeded.  

These constraints are designed to ensure that the optimization respects both passenger demand and operational capacity limitations, leading to a feasible and realistic solution.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 1 Database Schema
-- Objective: Schema changes include creating tables for operational costs and aircraft capacities, modifying the airport table to better align with constraints, and adding configuration logic for scalar parameters and formulas.

CREATE TABLE aircraft_capacity (
  aircraft_type STRING,
  passenger_capacity INTEGER
);

CREATE TABLE operational_costs (
  aircraft_type STRING,
  airport_code STRING,
  cost FLOAT,
  aircraft_count INTEGER
);

CREATE TABLE airport (
  airport_code STRING,
  passenger_demand INTEGER,
  aircraft_movements INTEGER
);
```

### Data Dictionary  
- **aircraft_capacity**:  
  - **aircraft_type**: Identifies the type of aircraft.  
  - **passenger_capacity**: Represents the maximum number of passengers the aircraft can carry, used to ensure passenger demand is met.  

- **operational_costs**:  
  - **aircraft_type**: Identifies the type of aircraft.  
  - **airport_code**: Identifies the airport.  
  - **cost**: Represents the operational cost of assigning the aircraft to the airport, used in the objective function.  
  - **aircraft_count**: Represents the number of aircraft assigned, which is the decision variable in the optimization.  

- **airport**:  
  - **airport_code**: Identifies the airport.  
  - **passenger_demand**: Represents the total passenger demand at the airport, used in the passenger demand constraint.  
  - **aircraft_movements**: Represents the maximum number of aircraft movements at the airport, used in the aircraft movement constraint.  


### Retrieved Values

**Query 1: This is crucial for the objective function, as it provides the cost coefficients that need to be minimized.**

```sql
SELECT aircraft_type, airport_code, cost FROM operational_costs;
```

**Results (CSV format):**
```csv
aircraft_type,airport_code,cost
Boeing 737,JFK,5000.0
Airbus A320,JFK,5500.0
Boeing 777,JFK,8000.0
Boeing 737,LAX,4800.0
Airbus A320,LAX,5200.0
```

**Query 2: This data is essential for the passenger demand constraint, ensuring that the total capacity of assigned aircraft meets or exceeds the airport's demand.**

```sql
SELECT aircraft_type, passenger_capacity FROM aircraft_capacity;
```

**Results (CSV format):**
```csv
aircraft_type,passenger_capacity
Boeing 737,150
Airbus A320,180
Boeing 777,300
```

**Query 3: This data is necessary for both the passenger demand and aircraft movement constraints.**

```sql
SELECT airport_code, passenger_demand, aircraft_movements FROM airport;
```

**Results (CSV format):**
```csv
airport_code,passenger_demand,aircraft_movements
JFK,15000,50
LAX,20000,60
ORD,18000,55
```

**Query 4: This data is important for understanding the current state of aircraft assignments and for ensuring that the aircraft movement constraint is not exceeded.**

```sql
SELECT airport_code, SUM(aircraft_count) AS total_aircraft_count FROM operational_costs GROUP BY airport_code;
```

**Results (CSV format):**
```csv
airport_code,total_aircraft_count
JFK,0
LAX,0
```

**Query 5: This data helps in verifying whether the passenger demand constraint is being met.**

```sql
SELECT oc.airport_code, SUM(ac.passenger_capacity * oc.aircraft_count) AS total_passenger_capacity FROM operational_costs oc JOIN aircraft_capacity ac ON oc.aircraft_type = ac.aircraft_type GROUP BY oc.airport_code;
```

**Results (CSV format):**
```csv
airport_code,total_passenger_capacity
JFK,0
LAX,0
```

**Query 6: This data is useful for understanding the current operational costs and for identifying potential areas for cost reduction.**

```sql
SELECT airport_code, SUM(cost * aircraft_count) AS total_operational_cost FROM operational_costs GROUP BY airport_code;
```

**Results (CSV format):**
```csv
airport_code,total_operational_cost
JFK,0.0
LAX,0.0
```

**Query 7: This combined data is useful for detailed analysis and for ensuring that both the passenger demand and aircraft movement constraints are considered in the optimization.**

```sql
SELECT oc.aircraft_type, oc.airport_code, oc.cost, oc.aircraft_count, ac.passenger_capacity FROM operational_costs oc JOIN aircraft_capacity ac ON oc.aircraft_type = ac.aircraft_type;
```

**Results (CSV format):**
```csv
aircraft_type,airport_code,cost,aircraft_count,passenger_capacity
Boeing 737,JFK,5000.0,0,150
Airbus A320,JFK,5500.0,0,180
Boeing 777,JFK,8000.0,0,300
Boeing 737,LAX,4800.0,0,150
Airbus A320,LAX,5200.0,0,180
```

**Query 8: This data is critical for the aircraft movement constraint, ensuring that the total number of assigned aircraft does not exceed the airport's capacity.**

```sql
SELECT airport_code, aircraft_movements FROM airport;
```

**Results (CSV format):**
```csv
airport_code,aircraft_movements
JFK,50
LAX,60
ORD,55
```

**Query 9: This data is necessary for the passenger demand constraint, ensuring that the total capacity of assigned aircraft meets or exceeds the airport's demand.**

```sql
SELECT airport_code, passenger_demand FROM airport;
```

**Results (CSV format):**
```csv
airport_code,passenger_demand
JFK,15000
LAX,20000
ORD,18000
```

**Query 10: This data is useful for understanding the overall operational cost and for setting the optimization goal.**

```sql
SELECT SUM(cost * aircraft_count) AS total_operational_cost FROM operational_costs;
```

**Results (CSV format):**
```csv
total_operational_cost
0.0
```

