# Complete Optimization Problem and Solution: customers_card_transactions

## 1. Problem Context and Goals

### Context  
A bank is focused on reducing the total transaction fees incurred by its customers by strategically allocating transactions across different card types. Each card type has a distinct fee structure, and the bank aims to optimize these allocations to minimize costs. The bank must ensure that the total transaction amount per customer does not exceed their predefined account limits. This operational decision-making process involves determining the appropriate transaction amounts for each card type while adhering to customer-specific constraints.  

The business configuration includes key parameters such as the fee associated with each card type, which directly influences the total transaction fee calculation. Additionally, the maximum transaction amount allowed per customer serves as a critical constraint to ensure compliance with account limits. The total transaction fee is calculated by summing the product of the fee for each card type and the corresponding transaction amount allocated to that card. This approach ensures a linear relationship between the decision variables and the objective, avoiding any nonlinear complexities such as variable products or divisions.  

### Goals  
The primary goal of this optimization problem is to minimize the total transaction fees incurred by the bank’s customers. This is achieved by optimizing the allocation of transaction amounts across different card types, each with its own fee structure. Success is measured by the reduction in the total fee, which is directly influenced by the fees associated with each card type and the transaction amounts allocated to them. The optimization process ensures that the total transaction amount per customer remains within their account limits, maintaining operational feasibility while achieving cost efficiency.  

## 2. Constraints  

The optimization problem is subject to one key constraint: the total transaction amount allocated to all card types for a given customer must not exceed their account limit. This ensures that the bank’s operational decisions remain within the bounds of each customer’s financial capacity. The constraint is expressed as the sum of transaction amounts across all card types for a customer being less than or equal to their account limit. This linear constraint aligns with the business requirement of maintaining customer-specific transaction limits while optimizing fee costs.  

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 2 Database Schema
-- Objective: Added Transaction_Allocations table to map missing decision variables, updated business configuration logic to include scalar parameters and formulas, and ensured schema follows normalization principles.

CREATE TABLE Card_Fees (
  card_type_code STRING,
  fee FLOAT
);

CREATE TABLE Account_Limits (
  customer_id INTEGER,
  account_limit FLOAT
);

CREATE TABLE Transaction_Allocations (
  customer_id INTEGER,
  card_id STRING,
  transaction_amount FLOAT
);
```

### Data Dictionary  
- **Card_Fees**:  
  - **Business Purpose**: Stores the fee associated with each card type.  
  - **Optimization Role**: Provides the coefficients for the objective function.  
  - **Columns**:  
    - `card_type_code`: Represents the type of card (e.g., VISA, MASTERCARD).  
    - `fee`: The fee associated with the card type, used to calculate the total transaction fee.  

- **Account_Limits**:  
  - **Business Purpose**: Defines the maximum transaction amount allowed per customer.  
  - **Optimization Role**: Provides the bounds for the constraints.  
  - **Columns**:  
    - `customer_id`: Unique identifier for the customer.  
    - `account_limit`: The maximum transaction amount allowed for the customer.  

- **Transaction_Allocations**:  
  - **Business Purpose**: Tracks the amount of transaction allocated to each card type per customer.  
  - **Optimization Role**: Represents the decision variables in the optimization model.  
  - **Columns**:  
    - `customer_id`: Unique identifier for the customer.  
    - `card_id`: Unique identifier for the card.  
    - `transaction_amount`: The amount of transaction allocated to the card.  


### Retrieved Values

**Query 1: This data is crucial for determining the cost associated with each card type.**

```sql
SELECT card_type_code, fee FROM Card_Fees;
```

**Results (CSV format):**
```csv
card_type_code,fee
VISA,0.02
MASTERCARD,0.03
AMEX,0.035
```

**Query 2: This data ensures that the total transaction amount per customer does not exceed their predefined limits.**

```sql
SELECT customer_id, account_limit FROM Account_Limits;
```

**Results (CSV format):**
```csv
customer_id,account_limit
1,1000.0
2,1500.0
3,2000.0
```

**Query 3: This data is necessary to understand the current state of transaction allocations and to optimize them further.**

```sql
SELECT customer_id, card_id, transaction_amount FROM Transaction_Allocations;
```

**Results (CSV format):**
```csv
customer_id,card_id,transaction_amount
1,CARD1,100.0
2,CARD2,200.0
3,CARD3,300.0
```

**Query 4: This aggregated data is important for verifying the constraint that the sum of transaction amounts across all card types for a customer must be less than or equal to their account limit.**

```sql
SELECT customer_id, SUM(transaction_amount) AS total_transaction_amount FROM Transaction_Allocations GROUP BY customer_id;
```

**Results (CSV format):**
```csv
customer_id,total_transaction_amount
1,100.0
2,200.0
3,300.0
```

**Query 5: This data is essential for calculating the current total transaction fee and for optimizing it further.**

```sql
SELECT ta.customer_id, SUM(ta.transaction_amount * cf.fee) AS total_transaction_fee FROM Transaction_Allocations ta JOIN Card_Fees cf ON ta.card_id = cf.card_type_code GROUP BY ta.customer_id;
```

**Results (CSV format):**
```csv
customer_id,total_transaction_fee
```

**Query 6: This data helps in understanding the cost distribution across different card types and identifying potential areas for optimization.**

```sql
SELECT cf.card_type_code, SUM(ta.transaction_amount * cf.fee) AS total_transaction_fee FROM Transaction_Allocations ta JOIN Card_Fees cf ON ta.card_id = cf.card_type_code GROUP BY cf.card_type_code;
```

**Results (CSV format):**
```csv
card_type_code,total_transaction_fee
```

**Query 7: This data provides insights into the usage patterns of different card types and helps in optimizing their allocation.**

```sql
SELECT card_id, SUM(transaction_amount) AS total_transaction_amount FROM Transaction_Allocations GROUP BY card_id;
```

**Results (CSV format):**
```csv
card_id,total_transaction_amount
CARD1,100.0
CARD2,200.0
CARD3,300.0
```

**Query 8: This data is crucial for verifying that the total transaction amount per customer does not exceed their account limit and for calculating the total transaction fee.**

```sql
SELECT ta.customer_id, SUM(ta.transaction_amount) AS total_transaction_amount, SUM(ta.transaction_amount * cf.fee) AS total_transaction_fee, al.account_limit FROM Transaction_Allocations ta JOIN Card_Fees cf ON ta.card_id = cf.card_type_code JOIN Account_Limits al ON ta.customer_id = al.customer_id GROUP BY ta.customer_id, al.account_limit;
```

**Results (CSV format):**
```csv
customer_id,total_transaction_amount,total_transaction_fee,account_limit
```

