# Complete Optimization Problem and Solution: debate

## 1. Problem Context and Goals

### Context  
A debate organization is focused on maximizing audience engagement by strategically assigning participants to debates. Participants come from various districts and political parties, and the organization aims to ensure diversity and fairness in representation. The key operational decision involves assigning participants to specific debates and sides (Affirmative or Negative) while adhering to constraints such as limiting the number of debates each participant can join and ensuring a balance between sides in each debate.  

The organization uses the number of audience members for each debate as a weighting factor to prioritize debates with higher engagement. Additionally, participant age is considered to limit the number of debates they can participate in, ensuring fairness and preventing overcommitment. The business configuration includes scalar parameters such as the typical audience size for debates and a limit on the number of debates a participant can join. These parameters are critical for weighting the objective function and enforcing constraints.  

### Goals  
The primary goal is to maximize total audience engagement across all debates. This is achieved by assigning participants to debates in a way that prioritizes debates with larger audiences. Success is measured by the weighted sum of audience sizes for the debates to which participants are assigned. The optimization process ensures that the assignments respect constraints on participant availability, side balance, and age-based limitations.  

## 2. Constraints  

The optimization problem must adhere to the following constraints:  
1. **Participant Limit**: Each participant can be assigned to no more than one debate. This ensures fairness and prevents overcommitment.  
2. **Side Balance**: For each debate, the number of participants assigned to the Affirmative side must equal the number assigned to the Negative side. This ensures fairness and balance in the debate structure.  
3. **Minimum Participation**: Each debate must have at least one participant assigned to it. This ensures that all debates are conducted.  
4. **Age-Based Constraint**: The number of debates a participant can join is limited by their age, specifically by dividing their age by 25. This ensures that younger participants are not overburdened.  

These constraints are designed to maintain fairness, balance, and operational feasibility while aligning with the linear structure of the optimization problem.  

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 2 Database Schema
-- Objective: Schema changes include creating a new table for audience size, updating business configuration logic to include missing scalar parameters, and ensuring all optimization requirements are mapped correctly.

CREATE TABLE participant_assignment (
  Debate_ID INTEGER,
  People_ID INTEGER,
  Side STRING
);

CREATE TABLE people (
  People_ID INTEGER,
  District STRING,
  Party STRING,
  Age INTEGER
);

CREATE TABLE audience_size (
  Debate_ID INTEGER,
  Num_of_Audience INTEGER
);
```

### Data Dictionary  
- **participant_assignment**: Represents the assignment of participants to debates on specific sides.  
  - *Debate_ID*: Identifier for the debate.  
  - *People_ID*: Identifier for the participant.  
  - *Side*: The side (Affirmative/Negative) the participant is assigned to.  
- **people**: Represents the participants in the debates.  
  - *People_ID*: Identifier for the participant.  
  - *District*: The district the participant represents.  
  - *Party*: The party the participant represents.  
  - *Age*: The age of the participant.  
- **audience_size**: Represents the number of audience members for each debate.  
  - *Debate_ID*: Identifier for the debate.  
  - *Num_of_Audience*: Number of audience members for the debate.  


### Retrieved Values

**Query 1: This data is crucial for the objective function, which aims to maximize total audience engagement.**

```sql
SELECT Debate_ID, Num_of_Audience FROM audience_size ORDER BY Num_of_Audience DESC;
```

**Results (CSV format):**
```csv
Debate_ID,Num_of_Audience
2,200
1,150
3,100
```

**Query 2: This data is needed to enforce the age-based constraint and ensure diversity in representation.**

```sql
SELECT People_ID, District, Party, Age FROM people;
```

**Results (CSV format):**
```csv
People_ID,District,Party,Age
101,District A,Party X,25
102,District B,Party Y,30
103,District C,Party Z,35
```

**Query 3: This data helps in understanding existing assignments and ensuring side balance in each debate.**

```sql
SELECT Debate_ID, People_ID, Side FROM participant_assignment;
```

**Results (CSV format):**
```csv
Debate_ID,People_ID,Side
1,101,Affirmative
1,102,Negative
2,103,Affirmative
```

**Query 4: This data is essential for enforcing the side balance constraint.**

```sql
SELECT Debate_ID, Side, COUNT(People_ID) AS Participant_Count FROM participant_assignment GROUP BY Debate_ID, Side;
```

**Results (CSV format):**
```csv
Debate_ID,Side,Participant_Count
1,Affirmative,1
1,Negative,1
2,Affirmative,1
```

**Query 5: This data is needed to enforce the age-based constraint (number of debates = age / 25).**

```sql
SELECT People_ID, FLOOR(Age / 25) AS Max_Debates FROM people;
```

**Results (CSV format):**
```csv
People_ID,Max_Debates
101,1
102,1
103,1
```

**Query 6: This data is important for ensuring the minimum participation constraint (each debate must have at least one participant).**

```sql
SELECT Debate_ID FROM audience_size WHERE Debate_ID NOT IN (SELECT DISTINCT Debate_ID FROM participant_assignment);
```

**Results (CSV format):**
```csv
Debate_ID
3
```

**Query 7: This data helps in ensuring that each debate has at least one participant and that participants are not overcommitted.**

```sql
SELECT Debate_ID, COUNT(People_ID) AS Total_Participants FROM participant_assignment GROUP BY Debate_ID;
```

**Results (CSV format):**
```csv
Debate_ID,Total_Participants
1,2
2,1
```

**Query 8: This data is crucial for enforcing the participant limit constraint (each participant can be assigned to no more than one debate).**

```sql
SELECT People_ID, COUNT(Debate_ID) AS Num_Debates FROM participant_assignment GROUP BY People_ID HAVING COUNT(Debate_ID) > 1;
```

**Results (CSV format):**
```csv
People_ID,Num_Debates
```

**Query 9: This data provides a baseline for measuring the success of the optimization process.**

```sql
SELECT SUM(Num_of_Audience) AS Total_Audience FROM audience_size;
```

**Results (CSV format):**
```csv
Total_Audience
450
```

**Query 10: This data helps in understanding the typical engagement level and can be used for weighting purposes.**

```sql
SELECT AVG(Num_of_Audience) AS Avg_Audience FROM audience_size;
```

**Results (CSV format):**
```csv
Avg_Audience
150.0
```

**Query 11: This data is important for ensuring diversity and fairness in representation.**

```sql
SELECT District, Party, COUNT(People_ID) AS Participant_Count FROM people GROUP BY District, Party;
```

**Results (CSV format):**
```csv
District,Party,Participant_Count
District A,Party X,1
District B,Party Y,1
District C,Party Z,1
```

**Query 12: This data is crucial for enforcing the participant limit constraint.**

```sql
SELECT People_ID, COUNT(Debate_ID) AS Num_Debates FROM participant_assignment GROUP BY People_ID;
```

**Results (CSV format):**
```csv
People_ID,Num_Debates
101,1
102,1
103,1
```

**Query 13: This data helps in ensuring side balance and minimum participation constraints.**

```sql
SELECT Debate_ID, Side, COUNT(People_ID) AS Participant_Count FROM participant_assignment GROUP BY Debate_ID, Side;
```

**Results (CSV format):**
```csv
Debate_ID,Side,Participant_Count
1,Affirmative,1
1,Negative,1
2,Affirmative,1
```

**Query 14: This data provides an overview of the available resources for the optimization process.**

```sql
SELECT COUNT(People_ID) AS Total_Participants FROM people;
```

**Results (CSV format):**
```csv
Total_Participants
3
```

**Query 15: This data helps in understanding the scale of the optimization problem.**

```sql
SELECT COUNT(Debate_ID) AS Total_Debates FROM audience_size;
```

**Results (CSV format):**
```csv
Total_Debates
3
```

