# Complete Optimization Problem and Solution: employee_hire_evaluation

## 1. Problem Context and Goals

### Context  
A retail chain is focused on optimizing its employee hiring and allocation strategy across multiple shops to minimize total hiring costs while ensuring each shop meets its staffing requirements. The business decisions involve determining the number of full-time and part-time employees to hire for each shop, as well as deciding which employees should receive bonuses based on their performance evaluations.  

The cost of hiring a full-time employee varies by shop, with each shop having a specific cost per full-time employee. Similarly, the cost of hiring a part-time employee also varies by shop. These costs are critical inputs to the optimization problem, as they directly influence the total hiring expenses.  

The business configuration includes the following operational parameters:  
- **Cost per full-time employee for each shop**: This represents the expense associated with hiring a full-time employee at a specific shop.  
- **Cost per part-time employee for each shop**: This represents the expense associated with hiring a part-time employee at a specific shop.  

The optimization problem is designed to ensure that each shop has sufficient staff to meet its operational needs while adhering to limits on the number of full-time and part-time employees that can be hired. The problem avoids scenarios that would require nonlinear relationships, such as variable products or divisions, ensuring a linear formulation.  

### Goals  
The primary goal of this optimization problem is to minimize the total hiring costs across all shops. This is achieved by carefully deciding the number of full-time and part-time employees to hire for each shop, taking into account the specific costs associated with each type of hire.  

Success is measured by the ability to reduce total hiring expenses while ensuring that each shop meets its minimum staffing requirements and does not exceed the maximum allowed number of full-time or part-time employees. The optimization goal is directly aligned with the operational parameters, ensuring a clear and linear relationship between the decisions and the objective.  

## 2. Constraints  

The optimization problem is subject to the following constraints, which ensure that the staffing decisions align with business requirements:  

1. **Minimum Staffing Requirement**: For each shop, the total number of full-time and part-time employees hired must meet or exceed the minimum staffing requirement. This ensures that each shop has sufficient staff to operate effectively.  

2. **Maximum Full-Time Employees**: For each shop, the number of full-time employees hired must not exceed the maximum allowed number of full-time employees. This ensures compliance with business policies regarding full-time staffing levels.  

3. **Maximum Part-Time Employees**: For each shop, the number of part-time employees hired must not exceed the maximum allowed number of part-time employees. This ensures compliance with business policies regarding part-time staffing levels.  

These constraints are designed to naturally lead to linear mathematical forms, avoiding any complexity that would require nonlinear relationships.  

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 3 Database Schema
-- Objective: Schema changes include creating a new table for objective coefficients (cost_full_time and cost_part_time) and updating business configuration logic to ensure all optimization requirements are mapped. Existing tables and configuration logic are reviewed for completeness and alignment with OR expert's analysis.

CREATE TABLE shop_staff_requirements (
  shop_id INTEGER,
  min_staff INTEGER,
  max_full_time INTEGER,
  max_part_time INTEGER
);

CREATE TABLE employee_bonus_decisions (
  employee_id INTEGER,
  y BOOLEAN
);

CREATE TABLE employee_hiring_decisions (
  shop_id INTEGER,
  x_full_time INTEGER,
  x_part_time INTEGER
);

CREATE TABLE employee_costs (
  shop_id INTEGER,
  cost_full_time INTEGER,
  cost_part_time INTEGER
);
```

### Data Dictionary  
The following tables and columns are critical to the optimization problem, with their business purposes and optimization roles clearly defined:  

- **shop_staff_requirements**:  
  - **shop_id**: Unique identifier for each shop.  
  - **min_staff**: Minimum number of employees required for the shop to operate effectively.  
  - **max_full_time**: Maximum number of full-time employees allowed for the shop.  
  - **max_part_time**: Maximum number of part-time employees allowed for the shop.  

- **employee_bonus_decisions**:  
  - **employee_id**: Unique identifier for each employee.  
  - **y**: Binary decision indicating whether a bonus is awarded to the employee.  

- **employee_hiring_decisions**:  
  - **shop_id**: Unique identifier for each shop.  
  - **x_full_time**: Number of full-time employees to hire for the shop.  
  - **x_part_time**: Number of part-time employees to hire for the shop.  

- **employee_costs**:  
  - **shop_id**: Unique identifier for each shop.  
  - **cost_full_time**: Cost per full-time employee for the shop.  
  - **cost_part_time**: Cost per part-time employee for the shop.  


### Retrieved Values

**Query 1: This data is crucial for the objective function, as it directly influences the total hiring costs.**

```sql
SELECT shop_id, cost_full_time, cost_part_time FROM employee_costs;
```

**Results (CSV format):**
```csv
shop_id,cost_full_time,cost_part_time
1,1500,800
2,1600,850
3,1700,900
```

**Query 2: This data is essential for the constraints, ensuring that each shop meets its staffing needs without exceeding the allowed limits.**

```sql
SELECT shop_id, min_staff, max_full_time, max_part_time FROM shop_staff_requirements;
```

**Results (CSV format):**
```csv
shop_id,min_staff,max_full_time,max_part_time
1,5,10,8
2,6,12,10
3,7,15,12
```

**Query 3: This data represents the decision variables that need to be optimized.**

```sql
SELECT shop_id, x_full_time, x_part_time FROM employee_hiring_decisions;
```

**Results (CSV format):**
```csv
shop_id,x_full_time,x_part_time
1,2,3
2,3,4
3,4,5
```

**Query 4: This aggregated data helps in evaluating the total hiring costs across all shops, which is the primary goal of the optimization.**

```sql
SELECT h.shop_id, h.x_full_time * c.cost_full_time AS total_full_time_cost, h.x_part_time * c.cost_part_time AS total_part_time_cost FROM employee_hiring_decisions h JOIN employee_costs c ON h.shop_id = c.shop_id;
```

**Results (CSV format):**
```csv
shop_id,total_full_time_cost,total_part_time_cost
1,3000,2400
2,4800,3400
3,6800,4500
```

**Query 5: This data is useful for verifying the minimum staffing requirement constraint.**

```sql
SELECT shop_id, x_full_time + x_part_time AS total_employees FROM employee_hiring_decisions;
```

**Results (CSV format):**
```csv
shop_id,total_employees
1,5
2,7
3,9
```

**Query 6: This summary statistic is critical for evaluating the overall objective of minimizing total hiring costs.**

```sql
SELECT SUM(h.x_full_time * c.cost_full_time + h.x_part_time * c.cost_part_time) AS total_hiring_cost FROM employee_hiring_decisions h JOIN employee_costs c ON h.shop_id = c.shop_id;
```

**Results (CSV format):**
```csv
total_hiring_cost
24900
```

**Query 7: This data is relevant if bonuses are part of the optimization problem, though it seems less central to the current problem context.**

```sql
SELECT employee_id, y FROM employee_bonus_decisions;
```

**Results (CSV format):**
```csv
employee_id,y
101,1
102,0
103,1
```

**Query 8: This combined view can be useful for a comprehensive analysis of hiring decisions and their associated costs.**

```sql
SELECT h.shop_id, h.x_full_time, h.x_part_time, c.cost_full_time, c.cost_part_time, h.x_full_time * c.cost_full_time AS total_full_time_cost, h.x_part_time * c.cost_part_time AS total_part_time_cost FROM employee_hiring_decisions h JOIN employee_costs c ON h.shop_id = c.shop_id;
```

**Results (CSV format):**
```csv
shop_id,x_full_time,x_part_time,cost_full_time,cost_part_time,total_full_time_cost,total_part_time_cost
1,2,3,1500,800,3000,2400
2,3,4,1600,850,4800,3400
3,4,5,1700,900,6800,4500
```

**Query 9: This combined view helps in verifying that the hiring decisions meet the staffing requirements and constraints.**

```sql
SELECT h.shop_id, h.x_full_time, h.x_part_time, s.min_staff, s.max_full_time, s.max_part_time FROM employee_hiring_decisions h JOIN shop_staff_requirements s ON h.shop_id = s.shop_id;
```

**Results (CSV format):**
```csv
shop_id,x_full_time,x_part_time,min_staff,max_full_time,max_part_time
1,2,3,5,10,8
2,3,4,6,12,10
3,4,5,7,15,12
```

