# Complete Optimization Problem and Solution: entrepreneur

## 1. Problem Context and Goals

### Context  
An investor is tasked with allocating a fixed budget across multiple entrepreneurs to maximize the total expected return on investment. The investor must decide how much money to allocate to each entrepreneur, ensuring that the total investment does not exceed the available budget and that no single entrepreneur receives more than a specified maximum amount. The available budget for investment is set at one million dollars, and the maximum amount that can be allocated to any single entrepreneur is two hundred thousand dollars. These parameters are designed to ensure risk management and diversification across the investment portfolio. The expected return rate for each entrepreneur is a key factor in determining the optimal allocation, as it directly influences the total return on investment.

### Goals  
The primary goal of this optimization problem is to maximize the total expected return on investment by strategically allocating the available budget across the entrepreneurs. Success is measured by the total return generated from the investments, which is calculated by multiplying the amount allocated to each entrepreneur by their respective expected return rate and summing these values across all entrepreneurs. The optimization process ensures that the allocation decisions are made in a way that respects the budget and individual investment limits while maximizing the overall return.

## 2. Constraints    

The optimization problem is subject to two key constraints:  
1. **Total Budget Constraint**: The sum of all investments allocated to the entrepreneurs must not exceed the total available budget of one million dollars. This ensures that the investor does not overspend and stays within the financial limits.  
2. **Individual Investment Limit**: The amount allocated to any single entrepreneur must not exceed two hundred thousand dollars. This constraint ensures that the investment portfolio remains diversified and that no single entrepreneur receives an disproportionately large share of the budget.  

These constraints are designed to align with the investor's risk management strategy and ensure that the allocation decisions are both financially sound and operationally feasible.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 1 Database Schema
-- Objective: Schema changes and configuration logic updates implemented to address missing optimization requirements and mapping gaps identified by the OR expert.

CREATE TABLE entrepreneurs (
  entrepreneur_id INTEGER,
  return_rate FLOAT,
  investment FLOAT
);
```

### Data Dictionary  
The `entrepreneurs` table contains the following columns, each with a specific business purpose and optimization role:  
- **entrepreneur_id**: A unique identifier for each entrepreneur, used to index the decision variables and constraints in the optimization problem.  
- **return_rate**: The expected return rate for each entrepreneur, which serves as the coefficient in the objective function to maximize the total return on investment.  
- **investment**: The amount of money allocated to each entrepreneur, which is the decision variable in the optimization problem.  


### Retrieved Values

**Query 1: This data is crucial as it provides the decision variables (investment) and the coefficients (return_rate) for the objective function.**

```sql
SELECT entrepreneur_id, return_rate, investment FROM entrepreneurs;
```

**Results (CSV format):**
```csv
entrepreneur_id,return_rate,investment
1,0.12,150000.0
2,0.18,200000.0
3,0.1,100000.0
```

**Query 2: This is important to ensure that the total budget constraint is respected and to understand the current allocation.**

```sql
SELECT SUM(investment) AS total_investment FROM entrepreneurs;
```

**Results (CSV format):**
```csv
total_investment
450000.0
```

**Query 3: This helps identify any entrepreneurs who are already receiving more than the maximum allowed investment, which is critical for enforcing the individual investment limit constraint.**

```sql
SELECT entrepreneur_id, investment FROM entrepreneurs WHERE investment > 200000;
```

**Results (CSV format):**
```csv
entrepreneur_id,investment
```

**Query 4: This data is useful for prioritizing investments in entrepreneurs who offer the highest potential return, which is key to maximizing the total expected return.**

```sql
SELECT entrepreneur_id, return_rate FROM entrepreneurs ORDER BY return_rate DESC;
```

**Results (CSV format):**
```csv
entrepreneur_id,return_rate
2,0.18
1,0.12
3,0.1
```

**Query 5: This provides an upper bound on the total expected return, which can be useful for benchmarking and understanding the potential of the optimization.**

```sql
SELECT SUM(return_rate * 200000) AS potential_total_return FROM entrepreneurs;
```

**Results (CSV format):**
```csv
potential_total_return
80000.0
```

**Query 6: This helps identify entrepreneurs who have not yet received significant investment, which could be an opportunity for reallocation within the budget constraints.**

```sql
SELECT entrepreneur_id, investment FROM entrepreneurs WHERE investment <= 0 OR investment IS NULL;
```

**Results (CSV format):**
```csv
entrepreneur_id,investment
```

**Query 7: This is essential for understanding how much of the budget is still available for further allocation.**

```sql
SELECT 1000000 - SUM(investment) AS remaining_budget FROM entrepreneurs;
```

**Results (CSV format):**
```csv
remaining_budget
550000.0
```

**Query 8: This data is useful for identifying entrepreneurs who can still receive additional investment without violating the individual investment limit constraint.**

```sql
SELECT entrepreneur_id, investment FROM entrepreneurs WHERE investment < 200000;
```

**Results (CSV format):**
```csv
entrepreneur_id,investment
1,150000.0
3,100000.0
```

**Query 9: This provides a baseline for the current total return, which can be compared against the optimized total return to measure the effectiveness of the optimization.**

```sql
SELECT SUM(return_rate * investment) AS current_total_return FROM entrepreneurs;
```

**Results (CSV format):**
```csv
current_total_return
64000.0
```

**Query 10: This helps prioritize investments in entrepreneurs who offer the best return on investment, which is critical for maximizing the total expected return.**

```sql
SELECT entrepreneur_id, return_rate / investment AS return_per_dollar FROM entrepreneurs WHERE investment > 0 ORDER BY return_per_dollar DESC;
```

**Results (CSV format):**
```csv
entrepreneur_id,return_per_dollar
3,1e-06
2,9e-07
1,8e-07
```

