# Complete Optimization Problem and Solution: local_govt_in_alabama

## 1. Problem Context and Goals

### Context  
The local government in Alabama is tasked with optimizing the allocation of participants to events to ensure the highest possible satisfaction among attendees while adhering to operational constraints. The decision to assign a participant to an event is represented as a binary choice: either the participant is assigned (1) or not (0). The goal is to maximize the total satisfaction score across all assignments, where the satisfaction score reflects how much a participant is expected to enjoy a specific event. 

Each event has a predefined capacity, which limits the number of participants that can be assigned to it. This capacity is a critical operational parameter, ensuring that events are not overcrowded and remain manageable. Additionally, each participant can be assigned to at most one event, ensuring fairness and preventing overcommitment. 

The satisfaction scores are derived from historical data and participant preferences, providing a realistic measure of expected enjoyment. The default event capacity is set to 50 participants, a typical size for local government events, ensuring manageable group sizes. This configuration ensures that the optimization problem remains linear, avoiding complex scenarios such as variable products or divisions.

### Goals  
The primary goal of this optimization problem is to maximize the overall satisfaction of participants by assigning them to events in the most effective way. This is achieved by summing the satisfaction scores of all assigned participant-event pairs. Success is measured by the total satisfaction score, which directly reflects the quality of the assignments. The optimization process ensures that the assignments respect event capacities and participant availability, leading to a fair and efficient allocation.

## 2. Constraints  

The optimization problem is subject to two key constraints:  
1. **Event Capacity Constraint**: The total number of participants assigned to any single event must not exceed the event's predefined capacity. This ensures that events remain within manageable limits and do not become overcrowded.  
2. **Participant Assignment Constraint**: Each participant can be assigned to at most one event. This ensures fairness and prevents participants from being overcommitted to multiple events simultaneously.  

These constraints are designed to align with the operational realities of event planning, ensuring that the solution is both practical and effective.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 1 Database Schema
-- Objective: Added tables for satisfaction scores and event capacities, updated Participants_in_Events table, and created business_configuration_logic.json for scalar parameters and formulas.

CREATE TABLE Satisfaction_Scores (
  Event_ID INTEGER,
  Participant_ID INTEGER,
  Score FLOAT
);

CREATE TABLE Event_Capacities (
  Event_ID INTEGER,
  Capacity INTEGER
);

CREATE TABLE Participants_in_Events (
  Event_ID INTEGER,
  Participant_ID INTEGER
);
```

### Data Dictionary  
- **Satisfaction_Scores**:  
  - **Business Purpose**: Captures the satisfaction score of each participant for each event, reflecting how much a participant is expected to enjoy a specific event.  
  - **Optimization Role**: Provides the coefficients for the objective function, representing the value of assigning a participant to an event.  
  - **Columns**:  
    - **Event_ID**: Unique identifier for the event, linking it to the optimization model.  
    - **Participant_ID**: Unique identifier for the participant, linking them to the optimization model.  
    - **Score**: The satisfaction score of the participant for the event, used as a coefficient in the objective function.  

- **Event_Capacities**:  
  - **Business Purpose**: Defines the maximum number of participants allowed in each event, ensuring events remain within manageable limits.  
  - **Optimization Role**: Provides the bounds for the event capacity constraints.  
  - **Columns**:  
    - **Event_ID**: Unique identifier for the event, linking it to the optimization model.  
    - **Capacity**: The maximum number of participants allowed in the event, used as a bound in the capacity constraint.  

- **Participants_in_Events**:  
  - **Business Purpose**: Represents the assignment of participants to events, capturing the decision to assign a participant to an event.  
  - **Optimization Role**: Represents the binary decision variables in the optimization model.  
  - **Columns**:  
    - **Event_ID**: Unique identifier for the event, linking it to the optimization model.  
    - **Participant_ID**: Unique identifier for the participant, linking them to the optimization model.  


### Retrieved Values

**Query 1: This data is crucial for the objective function, as it represents the coefficients to maximize.**

```sql
SELECT Event_ID, Participant_ID, Score FROM Satisfaction_Scores;
```

**Results (CSV format):**
```csv
Event_ID,Participant_ID,Score
1,101,0.85
1,102,0.75
1,103,0.65
2,101,0.7
2,102,0.9
2,103,0.8
3,101,0.6
3,102,0.85
3,103,0.95
```

**Query 2: This data is essential for the event capacity constraint, ensuring that the number of participants assigned to an event does not exceed its capacity.**

```sql
SELECT Event_ID, Capacity FROM Event_Capacities;
```

**Results (CSV format):**
```csv
Event_ID,Capacity
1,50
2,100
3,75
```

**Query 3: This data represents the decision variables in the optimization model, indicating whether a participant is assigned to an event.**

```sql
SELECT Event_ID, Participant_ID FROM Participants_in_Events;
```

**Results (CSV format):**
```csv
Event_ID,Participant_ID
1,101
1,102
1,103
2,101
2,102
2,103
3,101
3,102
3,103
```

**Query 4: This summary data helps in verifying the event capacity constraint by comparing the number of assigned participants to the event's capacity.**

```sql
SELECT Event_ID, COUNT(Participant_ID) AS Assigned_Participants FROM Participants_in_Events GROUP BY Event_ID;
```

**Results (CSV format):**
```csv
Event_ID,Assigned_Participants
1,3
2,3
3,3
```

**Query 5: This data can be useful for identifying the best possible event for each participant, which can inform initial assignments or provide insights into potential optimizations.**

```sql
SELECT Participant_ID, MAX(Score) AS Max_Satisfaction_Score FROM Satisfaction_Scores GROUP BY Participant_ID;
```

**Results (CSV format):**
```csv
Participant_ID,Max_Satisfaction_Score
101,0.85
102,0.9
103,0.95
```

**Query 6: This summary data can help identify events that generally provide higher satisfaction, which can be useful for prioritizing certain events in the optimization process.**

```sql
SELECT Event_ID, AVG(Score) AS Avg_Satisfaction_Score FROM Satisfaction_Scores GROUP BY Event_ID;
```

**Results (CSV format):**
```csv
Event_ID,Avg_Satisfaction_Score
1,0.75
2,0.7999999999999999
3,0.7999999999999999
```

**Query 7: This data is important for ensuring that all participants are considered in the optimization process and that no participant is left unassigned.**

```sql
SELECT COUNT(DISTINCT Participant_ID) AS Unassigned_Participants FROM Satisfaction_Scores WHERE Participant_ID NOT IN (SELECT Participant_ID FROM Participants_in_Events);
```

**Results (CSV format):**
```csv
Unassigned_Participants
0
```

**Query 8: This combined data can be useful for understanding the trade-offs between satisfaction and event capacity constraints.**

```sql
SELECT s.Event_ID, s.Participant_ID, s.Score, e.Capacity FROM Satisfaction_Scores s JOIN Event_Capacities e ON s.Event_ID = e.Event_ID;
```

**Results (CSV format):**
```csv
Event_ID,Participant_ID,Score,Capacity
1,101,0.85,50
1,102,0.75,50
1,103,0.65,50
2,101,0.7,100
2,102,0.9,100
2,103,0.8,100
3,101,0.6,75
3,102,0.85,75
3,103,0.95,75
```

**Query 9: This data can be used to prioritize assignments for participants who are currently unassigned, ensuring they are considered in the optimization process.**

```sql
SELECT s.Event_ID, s.Participant_ID, s.Score FROM Satisfaction_Scores s WHERE s.Participant_ID NOT IN (SELECT Participant_ID FROM Participants_in_Events);
```

**Results (CSV format):**
```csv
Event_ID,Participant_ID,Score
```

**Query 10: This summary data helps in evaluating the current overall satisfaction for each event, which can be useful for assessing the effectiveness of current assignments.**

```sql
SELECT p.Event_ID, SUM(s.Score) AS Total_Satisfaction_Score FROM Participants_in_Events p JOIN Satisfaction_Scores s ON p.Event_ID = s.Event_ID AND p.Participant_ID = s.Participant_ID GROUP BY p.Event_ID;
```

**Results (CSV format):**
```csv
Event_ID,Total_Satisfaction_Score
1,2.25
2,2.4
3,2.4
```

