# Complete Optimization Problem and Solution: musical

## 1. Problem Context and Goals

### Context  
A theater company is planning multiple musical performances and aims to maximize audience engagement by optimally assigning actors to roles. The engagement metric is calculated as a weighted combination of each actor's age and their performance duration. The company has a set of actors, each with a specific age and performance duration, and a set of roles that need to be filled. The decision to assign an actor to a role is binary, meaning an actor is either assigned to a role or not. 

The company has established two weights to balance the importance of age and performance duration in the engagement metric. The weight for age is set to 0.6, emphasizing the value of experience, while the weight for performance duration is set to 0.4, ensuring a balanced consideration of performance length. Additionally, the company has set a maximum total duration limit of 100 minutes for all performances combined to ensure the event remains within a feasible timeframe.

The goal is to assign actors to roles in such a way that the total engagement is maximized, while ensuring that each actor is assigned to at most one role, each role is filled by exactly one actor, and the total duration of all performances does not exceed the allowed limit.

### Goals  
The primary optimization goal is to maximize the total audience engagement across all performances. This is achieved by assigning actors to roles in a way that maximizes the weighted sum of their ages and performance durations. Success is measured by the total engagement score, which is calculated using the predefined weights for age and performance duration. The solution must also respect the constraints on actor assignments and the total performance duration.

## 2. Constraints    

The optimization problem is subject to the following constraints:

1. **Actor Assignment Limit**: Each actor can be assigned to at most one role. This ensures that no actor is overcommitted and can focus on a single performance.
   
2. **Role Fulfillment**: Each role must be filled by exactly one actor. This ensures that all roles are covered and no role is left unassigned.

3. **Total Duration Limit**: The combined duration of all performances must not exceed the maximum allowed total duration of 100 minutes. This ensures that the event remains within the planned timeframe.

These constraints ensure that the assignment of actors to roles is both feasible and aligned with the company's operational requirements.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 1 Database Schema
-- Objective: Schema changes include creating tables for roles, actors, and assignments. Configuration logic updates include adding weights for engagement metric and maximum duration.

CREATE TABLE roles (
  role_id INTEGER,
  musical_id INTEGER,
  role_name STRING
);

CREATE TABLE actors (
  actor_id INTEGER,
  age INTEGER,
  performance_duration INTEGER
);

CREATE TABLE assignments (
  assignment_id INTEGER,
  actor_id INTEGER,
  role_id INTEGER,
  assignment_status BOOLEAN
);
```

### Data Dictionary  
- **Roles Table**: Contains information about the roles available in different musicals. Each role has a unique identifier, a musical identifier, and a role name.
  - **role_id**: Unique identifier for each role.
  - **musical_id**: Identifier for the musical the role belongs to.
  - **role_name**: Name of the role.

- **Actors Table**: Contains information about the actors, including their age and performance duration.
  - **actor_id**: Unique identifier for each actor.
  - **age**: Age of the actor, used in the engagement metric.
  - **performance_duration**: Duration of the actor's performance, used in the engagement metric.

- **Assignments Table**: Tracks the assignments of actors to roles. Each assignment has a unique identifier, an actor identifier, a role identifier, and a status indicating whether the actor is assigned to the role.
  - **assignment_id**: Unique identifier for each assignment.
  - **actor_id**: Identifier for the actor assigned.
  - **role_id**: Identifier for the role assigned.
  - **assignment_status**: Indicates if the actor is assigned to the role (True/False).


### Retrieved Values

**Query 1: This data is crucial for the objective function coefficients.**

```sql
SELECT actor_id, age, performance_duration FROM actors;
```

**Results (CSV format):**
```csv
actor_id,age,performance_duration
1,28,15
2,35,20
3,22,10
```

**Query 2: This data is important to ensure that each role is assigned exactly one actor.**

```sql
SELECT role_id, musical_id, role_name FROM roles;
```

**Results (CSV format):**
```csv
role_id,musical_id,role_name
1,1,Lead
2,1,Supporting
3,2,Chorus
```

**Query 3: This data is important for the actor assignment limit constraint.**

```sql
SELECT assignment_id, actor_id, role_id, assignment_status FROM assignments WHERE assignment_status = TRUE;
```

**Results (CSV format):**
```csv
assignment_id,actor_id,role_id,assignment_status
1,1,1,1
2,2,2,1
3,3,3,1
```

**Query 4: This data is crucial for the total duration limit constraint.**

```sql
SELECT SUM(a.performance_duration) AS total_duration FROM assignments AS ass JOIN actors AS a ON ass.actor_id = a.actor_id WHERE ass.assignment_status = TRUE;
```

**Results (CSV format):**
```csv
total_duration
45
```

**Query 5: This data is important for identifying available actors for new assignments.**

```sql
SELECT a.actor_id, a.age, a.performance_duration FROM actors AS a LEFT JOIN assignments AS ass ON a.actor_id = ass.actor_id AND ass.assignment_status = TRUE WHERE ass.assignment_id IS NULL;
```

**Results (CSV format):**
```csv
actor_id,age,performance_duration
```

**Query 6: This data is important for identifying roles that still need to be filled.**

```sql
SELECT r.role_id, r.musical_id, r.role_name FROM roles AS r LEFT JOIN assignments AS ass ON r.role_id = ass.role_id AND ass.assignment_status = TRUE WHERE ass.assignment_id IS NULL;
```

**Results (CSV format):**
```csv
role_id,musical_id,role_name
```

**Query 7: This data is crucial for the objective function to maximize total engagement.**

```sql
SELECT actor_id, (0.6 * age) + (0.4 * performance_duration) AS engagement_score FROM actors;
```

**Results (CSV format):**
```csv
actor_id,engagement_score
1,22.8
2,29.0
3,17.2
```

**Query 8: This data is important for ensuring that all roles are assigned exactly one actor.**

```sql
SELECT COUNT(*) AS total_roles FROM roles;
```

**Results (CSV format):**
```csv
total_roles
3
```

**Query 9: This data is important for ensuring that there are enough actors to fill all roles.**

```sql
SELECT COUNT(*) AS total_actors FROM actors;
```

**Results (CSV format):**
```csv
total_actors
3
```

**Query 10: This data is important for understanding the current state of assignments.**

```sql
SELECT COUNT(*) AS assigned_actors FROM assignments WHERE assignment_status = TRUE;
```

**Results (CSV format):**
```csv
assigned_actors
3
```

**Query 11: This data is important for understanding the current state of assignments.**

```sql
SELECT COUNT(*) AS assigned_roles FROM assignments WHERE assignment_status = TRUE;
```

**Results (CSV format):**
```csv
assigned_roles
3
```

