# Complete Optimization Problem and Solution: product_catalog

## 1. Problem Context and Goals

### Context  
The company operates in a global market, selling products priced in three major currencies: US dollars, euros, and British pounds. To maximize revenue, the company needs to optimize the pricing of its products in each currency while ensuring prices remain within acceptable ranges and preventing arbitrage opportunities. Arbitrage could arise if price differences between currencies exceed a specified threshold, allowing customers to exploit discrepancies for profit.  

The company has established minimum and maximum price limits for each product in each currency to maintain competitiveness and profitability. Additionally, exchange rates between currencies are factored into the pricing strategy to ensure consistency across markets. The exchange rates currently in use are:  
- **Dollars to Euros**: 0.92  
- **Dollars to Pounds**: 0.79  
- **Euros to Pounds**: 0.86  

The maximum allowed price difference between currencies is set at $5.00 to prevent arbitrage. This ensures that prices in different currencies remain aligned after accounting for exchange rate fluctuations.  

The optimization problem focuses on adjusting product prices in dollars, euros, and pounds to maximize total revenue. Revenue is calculated as the sum of the product of each product's price in a currency and its expected sales volume in that currency. The decision variables are the prices in each currency, which can vary continuously within the defined bounds.  

### Goals  
The primary goal is to maximize total revenue across all products and currencies. Revenue is calculated by multiplying each product's price in dollars, euros, and pounds by its expected sales volume and summing these values across all products. Success is measured by achieving the highest possible revenue while adhering to the following constraints:  
1. Prices must remain within the specified minimum and maximum limits for each currency.  
2. Price differences between currencies, after applying exchange rates, must not exceed the maximum allowed difference of $5.00.  

This optimization ensures that the company's pricing strategy is both profitable and consistent across global markets, avoiding scenarios that could lead to arbitrage or customer dissatisfaction.  

## 2. Constraints  

The optimization problem is subject to the following constraints:  
1. **Price Bounds**:  
   - Each product's price in dollars must be between its minimum and maximum acceptable price in dollars.  
   - Each product's price in euros must be between its minimum and maximum acceptable price in euros.  
   - Each product's price in pounds must be between its minimum and maximum acceptable price in pounds.  

2. **Currency Price Differences**:  
   - The difference between a product's price in dollars converted to euros and its price in euros must not exceed the maximum allowed price difference of $5.00.  
   - The difference between a product's price in dollars converted to pounds and its price in pounds must not exceed the maximum allowed price difference of $5.00.  
   - The difference between a product's price in euros converted to pounds and its price in pounds must not exceed the maximum allowed price difference of $5.00.  

These constraints ensure that prices remain competitive, profitable, and consistent across currencies, preventing arbitrage opportunities.  

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 1 Database Schema
-- Objective: Schema changes include creating new tables for sales volumes and price constraints, modifying the Catalog_Contents table to include foreign keys, and adding business configuration logic for exchange rates and maximum price differences.

CREATE TABLE Product_Sales_Volume (
  product_id INTEGER,
  sales_volume INTEGER
);

CREATE TABLE Product_Price_Constraints (
  product_id INTEGER,
  min_price_dollars FLOAT,
  max_price_dollars FLOAT,
  min_price_euros FLOAT,
  max_price_euros FLOAT,
  min_price_pounds FLOAT,
  max_price_pounds FLOAT
);

CREATE TABLE Catalog_Contents (
  product_id INTEGER,
  price_in_dollars FLOAT,
  price_in_euros FLOAT,
  price_in_pounds FLOAT
);
```

### Data Dictionary  
- **Product_Sales_Volume**:  
  - **Purpose**: Stores the expected sales volume for each product.  
  - **Optimization Role**: Provides coefficients for the revenue calculation in the objective function.  
  - **Columns**:  
    - `product_id`: Unique identifier for each product.  
    - `sales_volume`: Expected sales volume for the product.  

- **Product_Price_Constraints**:  
  - **Purpose**: Defines the minimum and maximum acceptable prices for each product in dollars, euros, and pounds.  
  - **Optimization Role**: Provides bounds for the price constraints.  
  - **Columns**:  
    - `product_id`: Unique identifier for each product.  
    - `min_price_dollars`: Minimum acceptable price in dollars.  
    - `max_price_dollars`: Maximum acceptable price in dollars.  
    - `min_price_euros`: Minimum acceptable price in euros.  
    - `max_price_euros`: Maximum acceptable price in euros.  
    - `min_price_pounds`: Minimum acceptable price in pounds.  
    - `max_price_pounds`: Maximum acceptable price in pounds.  

- **Catalog_Contents**:  
  - **Purpose**: Stores the current prices of products in dollars, euros, and pounds.  
  - **Optimization Role**: Contains the decision variables for the optimization problem.  
  - **Columns**:  
    - `product_id`: Unique identifier for each product.  
    - `price_in_dollars`: Price of the product in dollars.  
    - `price_in_euros`: Price of the product in euros.  
    - `price_in_pounds`: Price of the product in pounds.  


### Retrieved Values

**Query 1: This data is crucial as it represents the decision variables that need to be optimized.**

```sql
SELECT product_id, price_in_dollars, price_in_euros, price_in_pounds FROM Catalog_Contents;
```

**Results (CSV format):**
```csv
product_id,price_in_dollars,price_in_euros,price_in_pounds
1,25.0,23.0,20.0
2,30.0,28.0,25.0
3,35.0,32.0,30.0
```

**Query 2: This data is essential for calculating the revenue, which is the objective function to be maximized.**

```sql
SELECT product_id, sales_volume FROM Product_Sales_Volume;
```

**Results (CSV format):**
```csv
product_id,sales_volume
1,150
2,200
3,100
```

**Query 3: This data is necessary to ensure that the optimized prices remain within acceptable bounds.**

```sql
SELECT product_id, min_price_dollars, max_price_dollars, min_price_euros, max_price_euros, min_price_pounds, max_price_pounds FROM Product_Price_Constraints;
```

**Results (CSV format):**
```csv
product_id,min_price_dollars,max_price_dollars,min_price_euros,max_price_euros,min_price_pounds,max_price_pounds
1,10.0,50.0,9.0,45.0,8.0,40.0
2,15.0,55.0,14.0,50.0,13.0,45.0
3,20.0,60.0,19.0,55.0,18.0,50.0
```

**Query 4: This data is useful for calculating the current revenue and understanding the relationship between price and sales volume.**

```sql
SELECT cc.product_id, cc.price_in_dollars, cc.price_in_euros, cc.price_in_pounds, psv.sales_volume FROM Catalog_Contents cc JOIN Product_Sales_Volume psv ON cc.product_id = psv.product_id;
```

**Results (CSV format):**
```csv
product_id,price_in_dollars,price_in_euros,price_in_pounds,sales_volume
1,25.0,23.0,20.0,150
2,30.0,28.0,25.0,200
3,35.0,32.0,30.0,100
```

**Query 5: This data is important to ensure that the optimization process respects the minimum and maximum price limits.**

```sql
SELECT cc.product_id, cc.price_in_dollars, cc.price_in_euros, cc.price_in_pounds, ppc.min_price_dollars, ppc.max_price_dollars, ppc.min_price_euros, ppc.max_price_euros, ppc.min_price_pounds, ppc.max_price_pounds FROM Catalog_Contents cc JOIN Product_Price_Constraints ppc ON cc.product_id = ppc.product_id;
```

**Results (CSV format):**
```csv
product_id,price_in_dollars,price_in_euros,price_in_pounds,min_price_dollars,max_price_dollars,min_price_euros,max_price_euros,min_price_pounds,max_price_pounds
1,25.0,23.0,20.0,10.0,50.0,9.0,45.0,8.0,40.0
2,30.0,28.0,25.0,15.0,55.0,14.0,50.0,13.0,45.0
3,35.0,32.0,30.0,20.0,60.0,19.0,55.0,18.0,50.0
```

**Query 6: This comprehensive data set is useful for a holistic view of the optimization problem, combining decision variables, objective function coefficients, and constraints.**

```sql
SELECT cc.product_id, cc.price_in_dollars, cc.price_in_euros, cc.price_in_pounds, psv.sales_volume, ppc.min_price_dollars, ppc.max_price_dollars, ppc.min_price_euros, ppc.max_price_euros, ppc.min_price_pounds, ppc.max_price_pounds FROM Catalog_Contents cc JOIN Product_Sales_Volume psv ON cc.product_id = psv.product_id JOIN Product_Price_Constraints ppc ON cc.product_id = ppc.product_id;
```

**Results (CSV format):**
```csv
product_id,price_in_dollars,price_in_euros,price_in_pounds,sales_volume,min_price_dollars,max_price_dollars,min_price_euros,max_price_euros,min_price_pounds,max_price_pounds
1,25.0,23.0,20.0,150,10.0,50.0,9.0,45.0,8.0,40.0
2,30.0,28.0,25.0,200,15.0,55.0,14.0,50.0,13.0,45.0
3,35.0,32.0,30.0,100,20.0,60.0,19.0,55.0,18.0,50.0
```

**Query 7: This data is important for ensuring that the price differences between currencies do not exceed the maximum allowed difference of $5.00.**

```sql
SELECT product_id, price_in_dollars, price_in_euros, price_in_pounds, price_in_dollars * 0.92 AS price_in_dollars_to_euros, price_in_dollars * 0.79 AS price_in_dollars_to_pounds, price_in_euros * 0.86 AS price_in_euros_to_pounds FROM Catalog_Contents;
```

**Results (CSV format):**
```csv
product_id,price_in_dollars,price_in_euros,price_in_pounds,price_in_dollars_to_euros,price_in_dollars_to_pounds,price_in_euros_to_pounds
1,25.0,23.0,20.0,23.0,19.75,19.78
2,30.0,28.0,25.0,27.6,23.700000000000003,24.08
3,35.0,32.0,30.0,32.2,27.650000000000002,27.52
```

**Query 8: This data is crucial for enforcing the constraint that price differences between currencies must not exceed $5.00.**

```sql
SELECT product_id, price_in_dollars, price_in_euros, price_in_pounds, price_in_dollars * 0.92 AS price_in_dollars_to_euros, price_in_dollars * 0.79 AS price_in_dollars_to_pounds, price_in_euros * 0.86 AS price_in_euros_to_pounds, 5.00 AS max_price_difference FROM Catalog_Contents;
```

**Results (CSV format):**
```csv
product_id,price_in_dollars,price_in_euros,price_in_pounds,price_in_dollars_to_euros,price_in_dollars_to_pounds,price_in_euros_to_pounds,max_price_difference
1,25.0,23.0,20.0,23.0,19.75,19.78,5.0
2,30.0,28.0,25.0,27.6,23.700000000000003,24.08,5.0
3,35.0,32.0,30.0,32.2,27.650000000000002,27.52,5.0
```

