# Complete Optimization Problem and Solution: real_estate_properties

## 1. Problem Context and Goals

### Context  
A real estate agency is focused on maximizing its total revenue from selling properties by determining the optimal selling price for each property. The agency must consider three key factors: the vendor's requested price, the buyer's offered price, and the property's features. The pricing model is designed to adjust the selling price linearly based on the property's features, ensuring that the final price reflects the property's value while remaining within the acceptable range for both the vendor and the buyer.

The agency uses a weighting factor to incorporate the influence of property features into the pricing model. This weighting factor, set at 0.3, ensures that features such as a pool, garage, or garden moderately influence the final selling price without overwhelming the base price. The agreed selling price for each property is calculated by adjusting the vendor's requested price based on the combined scores of the property's features, multiplied by the weighting factor.

The decision variable in this optimization problem is the agreed selling price for each property, which must be determined within the bounds set by the vendor's requested price and the buyer's offered price. The goal is to maximize the total revenue from all properties sold, ensuring that the pricing strategy is both fair and profitable.

### Goals  
The primary goal of this optimization problem is to maximize the total revenue generated from selling properties. This is achieved by determining the optimal agreed selling price for each property, ensuring that it falls within the acceptable range for both the vendor and the buyer. The success of this optimization is measured by the total sum of the agreed selling prices across all properties, which directly reflects the agency's revenue.

## 2. Constraints    

The optimization problem is subject to the following constraints:

1. **Vendor's Minimum Price Constraint**: The agreed selling price for each property must be at least as high as the vendor's requested price. This ensures that the vendor's minimum acceptable price is respected in the final agreement.

2. **Buyer's Maximum Price Constraint**: The agreed selling price for each property must not exceed the buyer's offered price. This ensures that the buyer's maximum willingness to pay is not surpassed.

3. **Feature-Adjusted Price Constraint**: The agreed selling price for each property is calculated by adjusting the vendor's requested price based on the property's features. This adjustment is made by multiplying the vendor's requested price by a factor that includes the weighting factor and the sum of the feature scores for the property. This ensures that the final price reflects the value added by the property's features.

These constraints ensure that the agreed selling price is both fair and realistic, balancing the interests of the vendor, the buyer, and the agency.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 1 Database Schema
-- Objective: Schema changes include adding a PropertyFeatures table to capture feature influence on pricing, and updating business_configuration_logic.json to include weighting factors for property features. The schema now supports the optimization model by incorporating property features into the pricing strategy.

CREATE TABLE Properties (
  property_id INTEGER,
  agreed_selling_price FLOAT,
  vendor_requested_price FLOAT,
  buyer_offered_price FLOAT
);

CREATE TABLE PropertyFeatures (
  property_id INTEGER,
  feature_name STRING,
  feature_score FLOAT
);
```

### Data Dictionary  
- **Properties Table**:  
  - **property_id**: A unique identifier for each property, used to link properties to their features and pricing details.  
  - **agreed_selling_price**: The final selling price agreed upon by the vendor and the buyer, which is the decision variable in the optimization model.  
  - **vendor_requested_price**: The minimum price the vendor is willing to accept, serving as the lower bound constraint in the optimization model.  
  - **buyer_offered_price**: The maximum price the buyer is willing to pay, serving as the upper bound constraint in the optimization model.  

- **PropertyFeatures Table**:  
  - **property_id**: A foreign key linking each feature to the corresponding property in the Properties table.  
  - **feature_name**: The name of the property feature, such as a pool, garage, or garden, which influences the selling price.  
  - **feature_score**: A score representing the influence of the feature on the selling price, used in the feature-adjusted price calculation.  


### Retrieved Values

**Query 1: Query Description: Retrieve the vendor's requested price, buyer's offered price, and the sum of feature scores for each property. This data is essential for calculating the agreed selling price within the constraints.**

```sql
SELECT p.property_id, p.vendor_requested_price, p.buyer_offered_price, SUM(pf.feature_score) AS total_feature_score FROM Properties p JOIN PropertyFeatures pf ON p.property_id = pf.property_id GROUP BY p.property_id, p.vendor_requested_price, p.buyer_offered_price;
```

**Results (CSV format):**
```csv
property_id,vendor_requested_price,buyer_offered_price,total_feature_score
1,300000.0,340000.0,0.25
2,450000.0,500000.0,0.2
3,240000.0,280000.0,0.1
```

**Query 2: Query Description: Retrieve the agreed selling price for each property. This is the decision variable that needs to be optimized to maximize total revenue.**

```sql
SELECT property_id, agreed_selling_price FROM Properties;
```

**Results (CSV format):**
```csv
property_id,agreed_selling_price
1,320000.0
2,480000.0
3,260000.0
```

**Query 3: Query Description: Retrieve the vendor's requested price and buyer's offered price for each property. These values define the lower and upper bounds for the agreed selling price.**

```sql
SELECT property_id, vendor_requested_price, buyer_offered_price FROM Properties;
```

**Results (CSV format):**
```csv
property_id,vendor_requested_price,buyer_offered_price
1,300000.0,340000.0
2,450000.0,500000.0
3,240000.0,280000.0
```

**Query 4: Query Description: Retrieve the feature scores for each property. These scores are used to adjust the vendor's requested price based on the property's features.**

```sql
SELECT property_id, feature_name, feature_score FROM PropertyFeatures;
```

**Results (CSV format):**
```csv
property_id,feature_name,feature_score
1,Pool,0.15
1,Garage,0.1
2,Garden,0.2
3,Garage,0.1
```

**Query 5: Query Description: Retrieve the total revenue from all properties based on the current agreed selling prices. This is the objective function that needs to be maximized.**

```sql
SELECT SUM(agreed_selling_price) AS total_revenue FROM Properties;
```

**Results (CSV format):**
```csv
total_revenue
1060000.0
```

**Query 6: Query Description: Retrieve properties where the agreed selling price is outside the acceptable range (below the vendor's requested price or above the buyer's offered price). This helps identify properties that violate the constraints.**

```sql
SELECT property_id, agreed_selling_price, vendor_requested_price, buyer_offered_price FROM Properties WHERE agreed_selling_price < vendor_requested_price OR agreed_selling_price > buyer_offered_price;
```

**Results (CSV format):**
```csv
property_id,agreed_selling_price,vendor_requested_price,buyer_offered_price
```

**Query 7: Query Description: Retrieve the average feature score across all properties. This can be useful for understanding the overall influence of features on pricing.**

```sql
SELECT AVG(feature_score) AS average_feature_score FROM PropertyFeatures;
```

**Results (CSV format):**
```csv
average_feature_score
0.1375
```

**Query 8: Query Description: Retrieve the maximum and minimum feature scores for each property. This helps in understanding the range of feature influences on pricing.**

```sql
SELECT property_id, MAX(feature_score) AS max_feature_score, MIN(feature_score) AS min_feature_score FROM PropertyFeatures GROUP BY property_id;
```

**Results (CSV format):**
```csv
property_id,max_feature_score,min_feature_score
1,0.15,0.1
2,0.2,0.2
3,0.1,0.1
```

**Query 9: Query Description: Retrieve the count of features for each property. This can be useful for understanding the complexity of each property's feature set.**

```sql
SELECT property_id, COUNT(feature_name) AS feature_count FROM PropertyFeatures GROUP BY property_id;
```

**Results (CSV format):**
```csv
property_id,feature_count
1,2
2,1
3,1
```

**Query 10: Query Description: Retrieve the properties with the highest and lowest feature scores. This helps identify properties with the most and least influential features.**

```sql
SELECT property_id, SUM(feature_score) AS total_feature_score FROM PropertyFeatures GROUP BY property_id ORDER BY total_feature_score DESC LIMIT 1;
```

**Results (CSV format):**
```csv
property_id,total_feature_score
1,0.25
```

**Query 11: Query Description: Retrieve the properties with the highest and lowest feature scores. This helps identify properties with the most and least influential features.**

```sql
SELECT property_id, SUM(feature_score) AS total_feature_score FROM PropertyFeatures GROUP BY property_id ORDER BY total_feature_score ASC LIMIT 1;
```

**Results (CSV format):**
```csv
property_id,total_feature_score
3,0.1
```

