# Complete PYOMO implementation - Retry Attempt 4

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def aircraft_assignment_optimization():
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    aircraft_types = ['Boeing 737', 'Airbus A320', 'Boeing 777']
    airports = ['JFK', 'LAX', 'ORD']
    
    # Operational costs (aircraft_type, airport_code)
    operational_costs = {
        ('Boeing 737', 'JFK'): 5000.0,
        ('Airbus A320', 'JFK'): 5500.0,
        ('Boeing 777', 'JFK'): 8000.0,
        ('Boeing 737', 'LAX'): 4800.0,
        ('Airbus A320', 'LAX'): 5200.0,
        ('Boeing 737', 'ORD'): 4900.0,
        ('Airbus A320', 'ORD'): 5300.0,
        ('Boeing 777', 'ORD'): 8100.0
    }
    
    # Passenger capacities (aircraft_type)
    passenger_capacities = {
        'Boeing 737': 150,
        'Airbus A320': 180,
        'Boeing 777': 300
    }
    
    # Airport demands and movements (airport_code)
    passenger_demands = {
        'JFK': 15000,
        'LAX': 20000,
        'ORD': 18000
    }
    
    aircraft_movements = {
        'JFK': 50,
        'LAX': 60,
        'ORD': 55
    }
    
    # 3. SETS
    model.aircraft_types = pyo.Set(initialize=aircraft_types)
    model.airports = pyo.Set(initialize=airports)
    
    # 4. PARAMETERS
    model.operational_costs = pyo.Param(model.aircraft_types, model.airports, initialize=operational_costs)
    model.passenger_capacities = pyo.Param(model.aircraft_types, initialize=passenger_capacities)
    model.passenger_demands = pyo.Param(model.airports, initialize=passenger_demands)
    model.aircraft_movements = pyo.Param(model.airports, initialize=aircraft_movements)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.aircraft_types, model.airports, within=pyo.NonNegativeIntegers)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.operational_costs[i, j] * model.x[i, j] for i in model.aircraft_types for j in model.airports)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.minimize)
    
    # 7. CONSTRAINTS
    # Passenger Demand Constraint
    def passenger_demand_rule(model, j):
        return sum(model.passenger_capacities[i] * model.x[i, j] for i in model.aircraft_types) >= model.passenger_demands[j]
    model.passenger_demand_constraint = pyo.Constraint(model.airports, rule=passenger_demand_rule)
    
    # Aircraft Movement Constraint
    def aircraft_movement_rule(model, j):
        return sum(model.x[i, j] for i in model.aircraft_types) <= model.aircraft_movements[j]
    model.aircraft_movement_constraint = pyo.Constraint(model.airports, rule=aircraft_movement_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    results = solver.solve(model, tee=True)
    
    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nAircraft Assignments:")
        for i in model.aircraft_types:
            for j in model.airports:
                x_val = pyo.value(model.x[i, j])
                if x_val > 0:
                    print(f"{i} at {j}: {int(x_val)}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Run the optimization
aircraft_assignment_optimization()